locals {
  owner = "jlindamo@twitch.tv"
  team  = "feeds"

  service = "graphdb"
}

provider "aws" {
  alias = "region1"
}

provider "aws" {
  alias = "region2"
}

variable "environment" {}

variable "security_groups" {
  description = "The security groups for the alb.  Usually .account.twitch_subnets_sg"
  type        = "map"
}

variable "forward_endpoints" {
  description = "Which URLs to send updates to"
  type        = "map"
}

variable "subnets" {
  description = "The VPC id for the alb target group.  Usually .account.vpc_id"
  type        = "map"
}

// --- setup permissions for dynamodb lambda

data "aws_iam_policy_document" "allow_lambda" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["lambda.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "iam_for_lambda" {
  name               = "${local.team}-${local.service}-${terraform.workspace}-lambda"
  assume_role_policy = "${data.aws_iam_policy_document.allow_lambda.json}"
  description        = "Allows lambda access to ${local.service} resources in ${terraform.workspace}"
}

resource "aws_iam_role_policy_attachment" "vpc_permissions" {
  role = "${aws_iam_role.iam_for_lambda.name}"

  // Built in policy that allows the lambda to be on the VPC
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy_attachment" "dynamo_permissions" {
  role = "${aws_iam_role.iam_for_lambda.name}"

  // Built in policy that allows the lambda to access dynamodb streams
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaDynamoDBExecutionRole"
}

// -- end dynamodb lambda permissions

module lambda_region1 {
  providers = {
    aws = "aws.region1"
  }

  source = "./region"

  environment      = "${var.environment}"
  security_groups  = "${var.security_groups["region1"]}"
  subnets          = "${var.subnets["region1"]}"
  forward_endpoint = "${var.forward_endpoints["region1"]}"
  role_arn         = "${aws_iam_role.iam_for_lambda.arn}"
}

module lambda_region2 {
  providers = {
    aws = "aws.region2"
  }

  source = "./region"

  environment      = "${var.environment}"
  security_groups  = "${var.security_groups["region2"]}"
  subnets          = "${var.subnets["region2"]}"
  forward_endpoint = "${var.forward_endpoints["region2"]}"
  role_arn         = "${aws_iam_role.iam_for_lambda.arn}"
}

output lamba_region1_arn {
  value = "${module.lambda_region1.lambda_arn}"
}

output lamba_region2_arn {
  value = "${module.lambda_region2.lambda_arn}"
}
