locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  service = "graphdb"

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  alarm_threshold_percents = {
    integration = 70
    staging     = 70
    production  = 70
  }

  alarm_threshold_percent = "${lookup(local.alarm_threshold_percents, terraform.workspace)}"

  alarm_periods = {
    integration = 60
    staging     = 60
    production  = 60
  }

  // Scale such that we use 50% of our maximum capacity
  target_utilization = 50

  alarm_period = "${lookup(local.alarm_periods, terraform.workspace)}"

  profile = "${lookup(local.profiles, terraform.workspace)}"
}

// You want read_capacity / 3 + write_capacity <= 1000
variable "edge_name" {
  type        = "string"
  description = "Which type of edge to create tables for"
}

variable "min_utilization" {
  description = "Minimum utilization the cluster is allowed to have, as a percent of max"
  default     = 0.05
}

variable "edge_read_capacity" {
  type        = "string"
  description = "How much read capacity for the edge"
}

variable "autoscale_role_arn" {
  type        = "string"
  description = "Which ARN allows autoscale events"
}

variable "list_to_read_capacity" {
  type        = "string"
  description = "How much read capacity for the reverse list_edges"
}

variable "edge_write_capacity" {
  type        = "string"
  description = "How much write capacity for the edge"
}

variable "count_read_capacity" {
  type        = "string"
  description = "How much read capacity for the count table"
}

variable "stream_lambda_arn" {}

variable "replica_region" {
  type        = "string"
  description = "Region that this table will be replicated in"
}

# For the edge
resource "aws_dynamodb_table" "edge_dynamodb" {
  name          = "${local.service}_${terraform.workspace}_${var.edge_name}"
  read_capacity = "${ceil(var.edge_read_capacity * __builtin_StringToFloat(var.min_utilization))}"

  // We multiply by 2 because the local_secondary_index write counts on the table's capacity
  write_capacity = "${ceil(var.edge_write_capacity * __builtin_StringToFloat(var.min_utilization) * 2)}"
  hash_key       = "fk"                                                                                  // from key
  range_key      = "to"                                                                                  // to entity

  attribute {
    name = "fk"
    type = "S"
  }

  attribute {
    name = "to"
    type = "S"
  }

  attribute {
    name = "tk" // to key
    type = "S"
  }

  attribute {
    name = "sk" // sort_key
    type = "N"
  }

  // Consistent reads allowed on LSI
  local_secondary_index {
    name            = "from_list"
    projection_type = "ALL"
    range_key       = "sk"
  }

  // Consistent reads not allowed on GSI
  global_secondary_index {
    name            = "to_list"
    projection_type = "ALL"
    hash_key        = "tk"
    range_key       = "sk"
    read_capacity   = "${ceil(var.list_to_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
    write_capacity  = "${ceil(var.edge_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  }

  lifecycle {
    prevent_destroy = true

    // Auto scale changes this.  We don't want to change it back
    // For why we have "global_secondary_index" see https://github.com/terraform-providers/terraform-provider-aws/issues/671
    // it is to avoid changing the capacities here
    ignore_changes = ["read_capacity", "write_capacity", "global_secondary_index"]
  }

  stream_enabled   = true
  stream_view_type = "NEW_AND_OLD_IMAGES"

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }

  point_in_time_recovery {
    enabled = true
  }

  replica {
    region_name = "${var.replica_region}"
  }
}

resource "aws_lambda_event_source_mapping" "dynamo_stream_events" {
  event_source_arn  = "${aws_dynamodb_table.edge_dynamodb.stream_arn}"
  function_name     = "${var.stream_lambda_arn}"
  starting_position = "TRIM_HORIZON"
}

module edge_dynamodb_autoscale {
  source             = "../../dynamodbautoscale"
  table_name         = "${aws_dynamodb_table.edge_dynamodb.name}"
  autoscale_role_arn = "${var.autoscale_role_arn}"
  min_read_capacity  = "${ceil(var.edge_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_read_capacity  = "${var.edge_read_capacity}"
  min_write_capacity = "${ceil(var.edge_write_capacity * __builtin_StringToFloat(var.min_utilization) * 2)}"
  max_write_capacity = "${var.edge_write_capacity * 2}"
  target_utilization = "${local.target_utilization}"
}

module edge_dynamodb_gsi_autoscale {
  source             = "../../dynamodbautoscale"
  index_name         = "to_list"
  table_name         = "${aws_dynamodb_table.edge_dynamodb.name}"
  autoscale_role_arn = "${var.autoscale_role_arn}"

  // See https://github.com/hashicorp/terraform/issues/14967 to explain __builtin_StringToFloat
  min_read_capacity  = "${ceil(var.list_to_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_read_capacity  = "${var.list_to_read_capacity}"
  min_write_capacity = "${ceil(var.edge_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_write_capacity = "${var.edge_write_capacity}"
  target_utilization = "${local.target_utilization}"
}

# For the counts for the edge
resource "aws_dynamodb_table" "edge_count_dynamodb" {
  name           = "${aws_dynamodb_table.edge_dynamodb.name}_count"
  read_capacity  = "${ceil(var.count_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  write_capacity = "${ceil(var.edge_write_capacity * __builtin_StringToFloat(var.min_utilization) * 2)}"
  hash_key       = "fr"                                                                                  // from
  range_key      = "et"                                                                                  // edge_type

  attribute {
    name = "fr" // from
    type = "S"
  }

  attribute {
    name = "et" // edge type
    type = "S"
  }

  lifecycle {
    prevent_destroy = true

    // Auto scale changes this.  We don't want to change it back
    // For why we have "global_secondary_index" see https://github.com/terraform-providers/terraform-provider-aws/issues/671
    // it is to avoid changing the capacities here
    ignore_changes = ["read_capacity", "write_capacity", "global_secondary_index"]
  }

  stream_enabled   = true
  stream_view_type = "NEW_AND_OLD_IMAGES"

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }

  point_in_time_recovery {
    enabled = true
  }

  replica {
    region_name = "${var.replica_region}"
  }
}

module edgecount_dynamodb_autoscale {
  source             = "../../dynamodbautoscale"
  table_name         = "${aws_dynamodb_table.edge_count_dynamodb.name}"
  autoscale_role_arn = "${var.autoscale_role_arn}"
  min_read_capacity  = "${ceil(var.count_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_read_capacity  = "${var.count_read_capacity}"
  min_write_capacity = "${ceil(var.edge_write_capacity * __builtin_StringToFloat(var.min_utilization) * 2)}"
  max_write_capacity = "${var.edge_write_capacity}"
  target_utilization = "${local.target_utilization}"
}

// ARN of the edge table
output edge_dynamodb_arn {
  value = "${aws_dynamodb_table.edge_dynamodb.arn}"
}

// ARN of the edge count table
output edge_count_dynamodb_arn {
  value = "${aws_dynamodb_table.edge_count_dynamodb.arn}"
}

// Name of the edge table
output edge_dynamodb_name {
  value = "${aws_dynamodb_table.edge_dynamodb.name}"
}

// Name of the edge count table
output edge_count_dynamodb_name {
  value = "${aws_dynamodb_table.edge_count_dynamodb.name}"
}
