locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  service = "graphdb"

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  alarm_threshold_percents = {
    integration = 70
    staging     = 70
    production  = 70
  }

  alarm_threshold_percent = "${lookup(local.alarm_threshold_percents, terraform.workspace)}"

  alarm_periods = {
    integration = 60
    staging     = 60
    production  = 60
  }

  // Scale such that we use 50% of our maximum capacity
  target_utilization = 50

  alarm_period = "${lookup(local.alarm_periods, terraform.workspace)}"

  profile = "${lookup(local.profiles, terraform.workspace)}"
}

variable "node_name" {
  type        = "string"
  description = "Which type of node to create tables for"
}

variable "min_utilization" {
  description = "Minimum utilization the cluster is allowed to have, as a percent of max"
  default     = 0.05
}

variable "stream_lambda_arn" {}

variable "node_read_capacity" {
  type        = "string"
  description = "How much read capacity for the node table"
}

variable "node_write_capacity" {
  type        = "string"
  description = "How much write capacity for the node table"
}

variable "list_read_capacity" {
  type        = "string"
  description = "How much read capacity for the list-nodes index"
}

variable "count_read_capacity" {
  type        = "string"
  description = "How much read capacity for the node counts"
}

variable "autoscale_role_arn" {
  type        = "string"
  description = "Which ARN allows autoscale events"
}

variable "replica_region" {
  type        = "string"
  description = "Region that this table will be replicated in"
}

# For the edge
resource "aws_dynamodb_table" "node_dynamodb" {
  name          = "${local.service}_${terraform.workspace}_n_${var.node_name}"
  read_capacity = "${ceil(var.node_read_capacity * __builtin_StringToFloat(var.min_utilization))}"

  write_capacity = "${ceil(var.node_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  hash_key       = "nk"                                                                             // node key (combination of nodetype + ":" + node_id

  attribute {
    name = "nk"
    type = "S"
  }

  attribute {
    name = "nt"
    type = "S"
  }

  attribute {
    name = "sk" // sort_key
    type = "N"
  }

  global_secondary_index {
    name            = "node_list"
    projection_type = "ALL"
    hash_key        = "nt"                                                                              // node type
    range_key       = "sk"                                                                              // sort key
    read_capacity   = "${ceil(var.list_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
    write_capacity  = "${ceil(var.node_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  }

  lifecycle {
    prevent_destroy = true

    // Auto scale changes this.  We don't want to change it back
    // For why we have "global_secondary_index" see https://github.com/terraform-providers/terraform-provider-aws/issues/671
    // it is to avoid changing the capacities here
    ignore_changes = ["read_capacity", "write_capacity", "global_secondary_index"]
  }

  stream_enabled   = true
  stream_view_type = "NEW_AND_OLD_IMAGES"

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }

  point_in_time_recovery {
    enabled = true
  }

  replica {
    region_name = "${var.replica_region}"
  }
}

resource "aws_lambda_event_source_mapping" "dynamo_stream_events" {
  event_source_arn  = "${aws_dynamodb_table.node_dynamodb.stream_arn}"
  function_name     = "${var.stream_lambda_arn}"
  starting_position = "TRIM_HORIZON"
}

module node_dynamodb_autoscale {
  source             = "../../dynamodbautoscale"
  table_name         = "${aws_dynamodb_table.node_dynamodb.name}"
  autoscale_role_arn = "${var.autoscale_role_arn}"

  min_read_capacity  = "${ceil(var.node_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_read_capacity  = "${var.node_read_capacity}"
  min_write_capacity = "${ceil(var.node_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_write_capacity = "${var.node_write_capacity}"
  target_utilization = "${local.target_utilization}"
}

module node_dynamodb_gsi_autoscale {
  source             = "../../dynamodbautoscale"
  index_name         = "node_list"
  table_name         = "${aws_dynamodb_table.node_dynamodb.name}"
  autoscale_role_arn = "${var.autoscale_role_arn}"

  // See https://github.com/hashicorp/terraform/issues/14967 to explain __builtin_StringToFloat
  min_read_capacity  = "${ceil(var.list_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_read_capacity  = "${var.list_read_capacity}"
  min_write_capacity = "${ceil(var.node_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_write_capacity = "${var.node_write_capacity}"
  target_utilization = "${local.target_utilization}"
}

# For the counts for the edge
resource "aws_dynamodb_table" "node_count_dynamodb" {
  name           = "${aws_dynamodb_table.node_dynamodb.name}_count"
  read_capacity  = "${ceil(var.count_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  write_capacity = "${ceil(var.node_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  hash_key       = "nt"                                                                              // node type

  attribute {
    name = "nt" // node type
    type = "S"
  }

  lifecycle {
    prevent_destroy = true

    // Auto scale changes this.  We don't want to change it back
    // For why we have "global_secondary_index" see https://github.com/terraform-providers/terraform-provider-aws/issues/671
    // it is to avoid changing the capacities here
    ignore_changes = ["read_capacity", "write_capacity", "global_secondary_index"]
  }

  stream_enabled   = true
  stream_view_type = "NEW_AND_OLD_IMAGES"

  tags {
    Environment = "${terraform.workspace}"
    Service     = "${local.service}"
    Team        = "${local.team}"
  }

  point_in_time_recovery {
    enabled = true
  }

  replica {
    region_name = "${var.replica_region}"
  }
}

module nodecount_dynamodb_autoscale {
  source             = "../../dynamodbautoscale"
  table_name         = "${aws_dynamodb_table.node_count_dynamodb.name}"
  autoscale_role_arn = "${var.autoscale_role_arn}"

  min_read_capacity  = "${ceil(var.count_read_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_read_capacity  = "${var.count_read_capacity}"
  min_write_capacity = "${ceil(var.node_write_capacity * __builtin_StringToFloat(var.min_utilization))}"
  max_write_capacity = "${var.node_write_capacity}"
  target_utilization = "${local.target_utilization}"
}

// ARN of the edge table
output node_dynamodb_arn {
  value = "${aws_dynamodb_table.node_dynamodb.arn}"
}

// ARN of the edge count table
output node_count_dynamodb_arn {
  value = "${aws_dynamodb_table.node_count_dynamodb.arn}"
}

// Name of the edge table
output node_dynamodb_name {
  value = "${aws_dynamodb_table.node_dynamodb.name}"
}

// Name of the edge count table
output node_count_dynamodb_name {
  value = "${aws_dynamodb_table.node_count_dynamodb.name}"
}
