resource "aws_ecs_service" "app" {
  name = "${var.team}-${var.service}"

  lifecycle {
    // Must ignore this so we can promote services to new task definitions
    // Must ignore desired count so the service can scale
    ignore_changes = ["task_definition", "desired_count"]
  }

  task_definition = "${aws_ecs_task_definition.app.family}"
  desired_count   = "${var.desired_count}"
  cluster         = "${data.aws_ecs_cluster.service_cluster.arn}"

  iam_role                           = "${var.service_iam_role}"
  deployment_maximum_percent         = 200
  deployment_minimum_healthy_percent = 90

  load_balancer {
    target_group_arn = "${var.target_group_arn}"
    container_name   = "${data.template_file.app_task_defs.vars.name}"
    container_port   = "${var.container_port}"
  }

  placement_strategy {
    type  = "spread"
    field = "attribute:ecs.availability-zone"
  }

  placement_strategy {
    type  = "spread"
    field = "instanceId"
  }
}

data "aws_ecs_cluster" service_cluster {
  cluster_name = "${var.cluster_name}"
}

# Placeholder task definition
data "template_file" "app_task_defs" {
  template = "${file("${path.module}/default_task_definition.json")}"

  vars {
    container_port = "${var.container_port}"
    name           = "${var.service}"
  }
}

resource "aws_ecs_task_definition" "app" {
  family                = "${var.team}-${var.service}-${var.environment}"
  task_role_arn         = "${var.task_role_arn}"
  container_definitions = "${data.template_file.app_task_defs.rendered}"
  network_mode          = "bridge"

  lifecycle {
    // Must ignore this so we can promote services to new task definitions
    ignore_changes = ["container_definitions"]
  }
}

# Output variables

output "service_name" {
  value = "${aws_ecs_service.app.name}"
}

output "task_family" {
  value = "${aws_ecs_task_definition.app.family}"
}
