# Task autoscaling
resource "aws_appautoscaling_target" "ecs_target" {
  max_capacity       = "${var.max_size}"
  min_capacity       = "${var.min_size}"
  resource_id        = "service/${var.cluster_name}/${var.service}"
  role_arn           = "${var.task_autoscale_iam_role}"
  scalable_dimension = "ecs:service:DesiredCount"
  service_namespace  = "ecs"

  lifecycle {
    // These tend to get changed from under us
    ignore_changes = ["role_arn"]
  }
}

resource "aws_appautoscaling_policy" "up" {
  name               = "${var.cluster_name}-${var.service}-scale-up"
  resource_id        = "${aws_appautoscaling_target.ecs_target.resource_id}"
  scalable_dimension = "ecs:service:DesiredCount"
  service_namespace  = "ecs"

  step_scaling_policy_configuration {
    adjustment_type         = "ChangeInCapacity"
    cooldown                = 300
    metric_aggregation_type = "Maximum"

    step_adjustment {
      metric_interval_lower_bound = 0
      scaling_adjustment          = 1
    }
  }

  depends_on = ["aws_appautoscaling_target.ecs_target"]
}

resource "aws_cloudwatch_metric_alarm" "cpu_high" {
  alarm_name          = "ecs-${var.cluster_name}-${var.service}-cpu-high"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  period              = "60"
  evaluation_periods  = "5"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  statistic           = "Maximum"
  threshold           = "${var.scale_up_cpu}"
  datapoints_to_alarm = "5"

  dimensions {
    ClusterName = "${var.cluster_name}"
    ServiceName = "${var.service}"
  }

  alarm_actions = ["${aws_appautoscaling_policy.up.arn}"]
}

resource "aws_appautoscaling_policy" "down" {
  name               = "${var.cluster_name}-${var.service}-scale-down"
  service_namespace  = "ecs"
  resource_id        = "${aws_appautoscaling_target.ecs_target.resource_id}"
  scalable_dimension = "ecs:service:DesiredCount"

  step_scaling_policy_configuration {
    adjustment_type         = "ChangeInCapacity"
    cooldown                = 300
    metric_aggregation_type = "Maximum"

    step_adjustment {
      metric_interval_upper_bound = 0
      scaling_adjustment          = -1
    }
  }

  depends_on = ["aws_appautoscaling_target.ecs_target"]
}

resource "aws_cloudwatch_metric_alarm" "cpu_low" {
  alarm_name          = "ecs-${var.cluster_name}-${var.service}-cpu-low"
  comparison_operator = "LessThanOrEqualToThreshold"
  period              = "60"
  evaluation_periods  = "5"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  statistic           = "Maximum"
  threshold           = "${var.scale_down_cpu}"
  datapoints_to_alarm = "5"

  dimensions {
    ClusterName = "${var.cluster_name}"
    ServiceName = "${var.service}"
  }

  alarm_actions = ["${aws_appautoscaling_policy.down.arn}"]
}
