package followsrpc

import (
	"fmt"

	"context"

	"golang.org/x/sync/errgroup"

	"github.com/twitchtv/twirp"
)

// MaxFollows is the maximum amount of users that a user can follow.
// This limit is used sometimes when loading followers without pagination.
const MaxFollows = 2000

// Subset of Follows client used by BulkCountFollowers
type followsCountClient interface {
	CountFollowers(context.Context, *UserIDReq) (*CountResp, error)
	CountFollows(context.Context, *UserIDReq) (*CountResp, error)
}

// BulkCountFollowers returns a map of userIDs to their number of followers.
//
// It makes one client.CountFollowers() request per userID provided.
// Requests are made concurrently.
//
// Doing this instead of implementing a BulkCount rpc method on the service
// helps reusing the cached responses (CountFollowers uses Cache-Control headers),
// as caching based on a list of IDs would be very ineficient.
func BulkCountFollowers(ctx context.Context, client followsCountClient, userIDs []string) (map[string]int32, error) {
	counts := make([]int32, len(userIDs))
	g, ctx := errgroup.WithContext(ctx)

	for i, userID := range userIDs {
		i, userID := i, userID // https://golang.org/doc/faq#closures_and_goroutines
		g.Go(func() (err error) {
			defer func() {
				if rErr := recover(); rErr != nil {
					err = twirp.InternalError(fmt.Sprintf("panic: %v", rErr))
				}
			}()

			resp, err := client.CountFollowers(ctx, &UserIDReq{UserId: userID})
			if err != nil {
				return err
			}
			if resp != nil {
				counts[i] = resp.Count
			}
			return nil
		})
	}

	if err := g.Wait(); err != nil {
		return nil, err
	}

	idsToCounts := map[string]int32{}
	for i, id := range userIDs {
		idsToCounts[id] = counts[i]
	}

	return idsToCounts, nil
}

// BulkCountFollows returns a map of userIDs to their number of follows.
//
// It makes one client.CountFollows() request per userID provided.
// Requests are made concurrently.
//
// Doing this instead of implementing a BulkCount() RPC method on the service
// helps reusing the cached responses (CountFollowers uses Cache-Control headers),
// as caching based on a list of IDs would be very ineficient.
func BulkCountFollows(ctx context.Context, client followsCountClient, userIDs []string) (map[string]int32, error) {
	counts := make([]int32, len(userIDs))
	g, ctx := errgroup.WithContext(ctx)

	for i, userID := range userIDs {
		i, userID := i, userID // https://golang.org/doc/faq#closures_and_goroutines
		g.Go(func() (err error) {
			defer func() {
				if rErr := recover(); rErr != nil {
					err = twirp.InternalError(fmt.Sprintf("panic: %v", rErr))
				}
			}()

			resp, err := client.CountFollows(ctx, &UserIDReq{UserId: userID})
			if err != nil {
				return err
			}
			if resp != nil {
				counts[i] = resp.Count
			}
			return nil
		})
	}

	if err := g.Wait(); err != nil {
		return nil, err
	}

	idsToCounts := map[string]int32{}
	for i, id := range userIDs {
		idsToCounts[id] = counts[i]
	}
	return idsToCounts, nil
}
