package dynamocursor

import (
	"fmt"

	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// Key is an object users can extract dynamodb attributes from
type Key interface {
	ExtractKey(item map[string]*dynamodb.AttributeValue) (string, error)
	AttrValue(v string) *dynamodb.AttributeValue
	AttrName() string
}

func newDynamodbKey(attrType string, name string) (Key, error) {
	switch attrType {
	case "N":
		return &numericDynamoKey{
			attrName: name,
		}, nil
	case "B":
		return &binaryDynamoKey{
			attrName: name,
		}, nil
	case "S":
		return &stringDynamoKey{
			attrName: name,
		}, nil
	}
	return nil, fmt.Errorf("invalid dynamodb key type %s", attrType)
}

type numericDynamoKey struct {
	attrName string
}

func (dk *numericDynamoKey) ExtractKey(item map[string]*dynamodb.AttributeValue) (string, error) {
	v, ok := item[dk.attrName]
	if !ok {
		return "", fmt.Errorf("extractKey: msg=missing key, key=%s, item=%v", dk.attrName, item)
	}
	return *v.N, nil
}

func (dk *numericDynamoKey) AttrName() string {
	return dk.attrName
}

func (dk *numericDynamoKey) AttrValue(v string) *dynamodb.AttributeValue {
	return &dynamodb.AttributeValue{
		N: &v,
	}
}

type stringDynamoKey struct {
	attrName string
}

func (dk *stringDynamoKey) ExtractKey(item map[string]*dynamodb.AttributeValue) (string, error) {
	v, ok := item[dk.attrName]
	if !ok {
		return "", fmt.Errorf("extractKey: msg=missing key, key=%s", dk.attrName)
	}
	return *v.S, nil
}

func (dk *stringDynamoKey) AttrName() string {
	return dk.attrName
}

func (dk *stringDynamoKey) AttrValue(v string) *dynamodb.AttributeValue {
	return &dynamodb.AttributeValue{
		S: &v,
	}
}

type binaryDynamoKey struct {
	attrName string
}

func (dk *binaryDynamoKey) ExtractKey(item map[string]*dynamodb.AttributeValue) (string, error) {
	v, ok := item[dk.attrName]
	if !ok {
		return "", fmt.Errorf("extractKey: msg=missing key, key=%s", dk.attrName)
	}
	return string(v.B), nil
}

func (dk *binaryDynamoKey) AttrValue(v string) *dynamodb.AttributeValue {
	return &dynamodb.AttributeValue{
		B: []byte(v),
	}
}

func (dk *binaryDynamoKey) AttrName() string {
	return dk.attrName
}
