package memcache

import (
	"bufio"
	"context"
	"errors"
	"io"
	"net"

	"code.justin.tv/hygienic/elastimemcache"
)

// Verifier allows checking that an endpoint is a valid memcache host
type Verifier struct {
	Dialer net.Dialer
	OnErr  func(error)
}

var _ elastimemcache.HealthVerifier = &Verifier{}

func (v *Verifier) check(err error) {
	if err != nil && v.OnErr != nil {
		v.OnErr(err)
	}
}

// CheckHealth returns true if the addr supports `version`
func (v *Verifier) CheckHealth(ctx context.Context, dst net.Addr) error {
	conn, err := v.Dialer.DialContext(ctx, dst.Network(), dst.String())
	if err != nil {
		return err
	}
	defer func() {
		v.check(conn.Close())
	}()
	if endTime, ok := ctx.Deadline(); ok {
		v.check(conn.SetDeadline(endTime))
	}
	_, err = io.WriteString(conn, "version\n")
	if err != nil {
		return err
	}
	r := bufio.NewReader(conn)
	lin, err := r.ReadSlice('\n')
	if err != nil {
		return err
	}
	if len(lin) == 0 {
		return errors.New("unable to find server version")
	}
	return nil
}
