package messagebatch

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
)

// Logger is the minimal interface for logging methods
type Logger interface {
	Log(keyvals ...interface{})
}

// Config controls how to configure your Batch message sender
type Config struct {
	Prefix        string
	FlushWaitTime *distconf.Duration
	FlushSize     *distconf.Int
}

// Load the configuration from distconf
func (c *Config) Load(d *distconf.Distconf) error {
	c.FlushSize = d.Int(c.Prefix+"flush_size", 1024)
	c.FlushWaitTime = d.Duration(c.Prefix+"flush_wait_time", time.Second)
	return nil
}

// Batcher allows sending batch messages
type Batcher struct {
	Log        Logger
	Config     *Config
	SendEvents func([]interface{}) error
	Events     chan interface{}
	onClose    chan struct{}
}

// Setup should be called before Start or Close
func (a *Batcher) Setup() error {
	a.onClose = make(chan struct{})
	return nil
}

// Event adds an event
func (a *Batcher) Event(event interface{}) {
	select {
	case a.Events <- event:
	default:
		a.Log.Log("event buffer full")
	}
}

// EmptyEvents empties out any leftover events and returns.  Returns early if ctx dies.
func (a *Batcher) EmptyEvents(ctx context.Context) error {
	events := make([]interface{}, 0, a.Config.FlushSize.Get())
	for {
		events = events[0:0]
		select {
		case event := <-a.Events:
			events = append(events, event)
		case <-ctx.Done():
			return ctx.Err()
		default:
			return nil
		}
		waitTime := time.After(a.Config.FlushWaitTime.Get())
	fillBuffer:
		for len(events) < int(a.Config.FlushSize.Get()) {
			select {
			case <-a.onClose:
				break fillBuffer
			case event := <-a.Events:
				events = append(events, event)
			case <-waitTime:
				break fillBuffer
			}
		}
		if err := a.SendEvents(events); err != nil {
			a.Log.Log("err", err, "unable to send events")
		}
	}
}

// Start sending added events.  Should run in a goroutine
func (a *Batcher) Start() error {
	events := make([]interface{}, 0, a.Config.FlushSize.Get())
	for {
		events = events[0:0]
		select {
		case <-a.onClose:
			return nil
		case event := <-a.Events:
			events = append(events, event)
		}
		waitTime := time.After(a.Config.FlushWaitTime.Get())
	fillBuffer:
		for len(events) < int(a.Config.FlushSize.Get()) {
			select {
			case <-a.onClose:
				break fillBuffer
			case event := <-a.Events:
				events = append(events, event)
			case <-waitTime:
				break fillBuffer
			}
		}
		if err := a.SendEvents(events); err != nil {
			a.Log.Log("err", err, "unable to send events")
		}
	}
}

// Close ends Start
func (a *Batcher) Close() error {
	close(a.onClose)
	return nil
}
