package objectcache

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
	"io"
)

// StreamDecoder allows custom decoding of objects from constructor functions
type StreamDecoder struct {
	Encode func(io.Writer) io.WriteCloser
	Decode func(io.Reader) (io.ReadCloser, error)
}

func newGzipWriter(w io.Writer) io.WriteCloser {
	return gzip.NewWriter(w)
}

func newGzipReader(reader io.Reader) (io.ReadCloser, error) {
	return gzip.NewReader(reader)
}

// GZipDecoder returns a StreamDecoder for objectcache that uses gzip to compress and decompress the JSON representation
// of the object.
func GZipDecoder() *StreamDecoder {
	return &StreamDecoder{
		Encode: newGzipWriter,
		Decode: newGzipReader,
	}
}

// Unmarshal will write data into *into* object.  It assumes into is a pointer, or nothing good will happen.
func (c *StreamDecoder) Unmarshal(data []byte, into interface{}) error {
	r, err := c.Decode(bytes.NewReader(data))
	if err != nil {
		return err
	}
	e := json.NewDecoder(r)
	if err := e.Decode(into); err != nil {
		return err
	}
	return r.Close()
}

// Marshal json encodes inside the custom encode function 'val' object.
func (c *StreamDecoder) Marshal(val interface{}) ([]byte, error) {
	var buf bytes.Buffer
	w := c.Encode(&buf)
	e := json.NewEncoder(w)
	err := e.Encode(val)
	if err != nil {
		return nil, err
	}
	if err := w.Close(); err != nil {
		return nil, err
	}
	return buf.Bytes(), nil
}
