package gomemcache

import (
	"context"
	"time"

	"code.justin.tv/hygienic/gomemcache/memcache"
	"code.justin.tv/hygienic/objectcache"
	"code.justin.tv/hygienic/statsdsender"
	"github.com/cep21/circuit"
)

// CacheClientPool allows caching objects into elasticache memcache
type CacheClientPool struct {
	Client  *memcache.Client
	Circuit *circuit.Circuit
}

// NewGomemcacheObjectPool is a helper function to create a gomemcache client
func NewGomemcacheObjectPool(client *memcache.Client, keyTTL func(string) time.Duration, keyPrefix string, Stats *statsdsender.ErrorlessStatSender, circuit *circuit.Circuit) *objectcache.ObjectCache {
	return &objectcache.ObjectCache{
		ClientPool: &CacheClientPool{
			Client:  client,
			Circuit: circuit,
		},
		Stats:     Stats,
		KeyPrefix: keyPrefix,
		KeyTTL:    keyTTL,
	}
}

var _ objectcache.CacheClientPool = &CacheClientPool{}
var _ objectcache.CacheClient = &CacheClientPool{}

// GetClient returns the cache pool itself: there is no extra client object
func (p *CacheClientPool) GetClient(ctx context.Context) objectcache.CacheClient {
	return p
}

// Delete tries an idempotent delete on the memcache client
func (p *CacheClientPool) Delete(ctx context.Context, key string) error {
	return p.Circuit.Run(ctx, func(ctx context.Context) error {
		err := p.Client.IdempotentDelete(ctx, key)
		if err == memcache.ErrCacheMiss {
			return nil
		}
		return err
	})
}

// Get returns the item and nil if ErrCacheMiss
func (p *CacheClientPool) Get(ctx context.Context, key string) ([]byte, error) {
	var item *memcache.Item
	err := p.Circuit.Run(ctx, func(ctx context.Context) error {
		var err error
		item, err = p.Client.Get(ctx, key)
		if err == memcache.ErrCacheMiss {
			return nil
		}
		return err
	})

	if err != nil {
		return nil, err
	}
	if item == nil {
		return nil, nil
	}
	return item.Value, nil
}

// Set the item into memcache
func (p *CacheClientPool) Set(ctx context.Context, key string, value []byte, ttl time.Duration) error {
	return p.Circuit.Run(ctx, func(ctx context.Context) error {
		return p.Client.Set(ctx, &memcache.Item{
			Key:        key,
			Value:      value,
			Expiration: int32(ttl.Seconds()),
		})
	})
}

// Return does nothing since *memcache.Client manages its own pool.
func (p *CacheClientPool) Return() {
}
