package objectcache

import (
	"context"
	"reflect"

	"code.justin.tv/hygienic/errors"
)

// MigrationObject is the supported wrapper interfaces needed for Migration
type MigrationObject interface {
	Invalidate(ctx context.Context, key string) error
	ForceCached(ctx context.Context, key string, vals interface{}) error
	Cached(ctx context.Context, key string, callback func() (i interface{}, e error), storeIntoPtr interface{}) error
}

var _ MigrationObject = &ObjectCache{}

// Migration is used when you need to move objects from an existing "From" cache to a "To" cache, but don't want to
// flood callbacks
type Migration struct {
	From MigrationObject
	To   MigrationObject
}

// Invalidate both From and To
func (r *Migration) Invalidate(ctx context.Context, key string) error {
	err1 := r.From.Invalidate(ctx, key)
	err2 := r.To.Invalidate(ctx, key)
	return errors.ConsolidateErrors([]error{err1, err2})
}

// ForceCached into both From and To
func (r *Migration) ForceCached(ctx context.Context, key string, vals interface{}) error {
	err1 := r.From.ForceCached(ctx, key, vals)
	err2 := r.To.ForceCached(ctx, key, vals)
	return errors.ConsolidateErrors([]error{err1, err2})
}

// Cached checks To for the key, if it does not exist, it checks From for the key.  The value is stored into
// both if it exists in neither, into To if it exists into From.  However, the value is *not* stored into From if it
// exists in To.
func (r *Migration) Cached(ctx context.Context, key string, callback func() (interface{}, error), storeIntoPtr interface{}) error {
	return r.To.Cached(ctx, key, func() (i interface{}, e error) {
		err := r.From.Cached(ctx, key, callback, storeIntoPtr)
		if err != nil {
			return nil, err
		}
		return reflect.ValueOf(storeIntoPtr).Elem().Interface(), nil
	}, storeIntoPtr)
}

// Ints is a Cached helper specifically for []int
func (r *Migration) Ints(ctx context.Context, key string, callback func() ([]int64, error)) ([]int64, error) {
	var storeIntoPtr []int64
	err := r.Cached(ctx, key, func() (interface{}, error) {
		return callback()
	}, &storeIntoPtr)
	if err != nil {
		return nil, err
	}
	return storeIntoPtr, nil
}
