package rlimit

import (
	"expvar"
	"syscall"
)

//rLimitVar reports process limits as returned by the rlimit syscall.
//
// NOTE: http://man7.org/linux/man-pages/man2/getrlimit.2.html
type rlimitVar struct {
	NoFile       syscall.Rlimit `json:"no_file,omitempty"`
	Stack        syscall.Rlimit `json:"stack,omitempty"`
	Core         syscall.Rlimit `json:"core,omitempty"`
	CPU          syscall.Rlimit `json:"cpu,omitempty"`
	FileSize     syscall.Rlimit `json:"file_size,omitempty"`
	Data         syscall.Rlimit `json:"data,omitempty"`
	AddressSpace syscall.Rlimit `json:"as,omitempty"`
	Error        string         `json:"error,omitempty"`
}

// Var returns the result of the getrlimit(2) syscall.
// Var's return value satisfies the expvar.Var interface and thus can be exposed on an expvar.Handler.
//
// Usage: f.serviceCommon.ExpvarHandler.Exported["rlimit"] = rlimit.Var()
func Var() expvar.Var {
	return expvar.Func(func() interface{} {
		return getRLimit()
	})
}

func getRLimit() rlimitVar {
	val := rlimitVar{}
	if err := syscall.Getrlimit(syscall.RLIMIT_NOFILE, &val.NoFile); err != nil {
		val.Error += " " + err.Error()
	}
	if err := syscall.Getrlimit(syscall.RLIMIT_STACK, &val.Stack); err != nil {
		val.Error += " " + err.Error()
	}
	if err := syscall.Getrlimit(syscall.RLIMIT_CORE, &val.Core); err != nil {
		val.Error += " " + err.Error()
	}
	if err := syscall.Getrlimit(syscall.RLIMIT_CPU, &val.CPU); err != nil {
		val.Error += " " + err.Error()
	}
	if err := syscall.Getrlimit(syscall.RLIMIT_FSIZE, &val.FileSize); err != nil {
		val.Error += " " + err.Error()
	}
	if err := syscall.Getrlimit(syscall.RLIMIT_DATA, &val.Data); err != nil {
		val.Error += " " + err.Error()
	}
	if err := syscall.Getrlimit(syscall.RLIMIT_AS, &val.AddressSpace); err != nil {
		val.Error += " " + err.Error()
	}
	return val
}
