package sqsextender

import (
	"context"

	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/sync/errgroup"
)

// MessageProcessor matches the sqsprocessor path
type MessageProcessor interface {
	Process(ctx context.Context, msg *sqs.Message) error
}

// ExtendedProcessor allows the sqs extender to work with the sqsprocessor api
type ExtendedProcessor struct {
	MessageProcessor          MessageProcessor
	OnPanic                   func(panicVal interface{})
	SQSMessageTimeoutExtender *SQSMessageTimeoutExtender
}

// Process extends the current message while also passing it to a MessageProcessor
func (e *ExtendedProcessor) Process(ctx context.Context, msg *sqs.Message) error {
	var panicVal interface{}
	extender := e.SQSMessageTimeoutExtender.Extend(msg.ReceiptHandle)
	eg, egCtx := errgroup.WithContext(ctx)
	eg.Go(func() error {
		return extender.Extend(egCtx)
	})
	eg.Go(func() error {
		defer extender.Stop()
		defer func() {
			if panicVal = recover(); panicVal != nil && e.OnPanic != nil {
				e.OnPanic(panicVal)
			}
		}()
		return e.MessageProcessor.Process(egCtx, msg)
	})
	ret := eg.Wait()
	if panicVal != nil {
		panic(panicVal)
	}
	return ret
}
