// Package workerpool allows you to queue async worker to execute later.  If you want to have it fall back to
// blocking behavior, you probably want to use OfferOrDo.  When you `Close` the pool, close blocks until worker tasks
// are drained, and the pool's API will still work: returning nil futures or allowing OfferOrDo to fall back to
// sync logic.
package workerpool

import (
	"context"
	"expvar"
	"fmt"
	"sync"
	"sync/atomic"
)

// Runnable is a function that can later be executed
type Runnable func() interface{}

// A worker drains futures to execute.
type worker struct {
	stop         chan struct{}
	done         chan struct{}
	futuresQueue <-chan *Future
}

// start should be called in a goroutine.  It will continue to pull futures that have been sent to the pool and execute
// them.  Start ends when the futures channel is finished draining.
func (p *worker) start() {
	defer close(p.done)
	for {
		// So we don't forever happen to pick p.ch below
		select {
		case <-p.stop:
			return
		default:
			// Pass.  Nobody wants to stop this worker
		}

		select {
		case f, isOk := <-p.futuresQueue:
			if !isOk {
				return
			}
			f.execute()
		case <-p.stop:
			return
		}
	}
}

// Stops new items from being added to the pool.  Drain will eventually end when all the items are processed
func (p *worker) Stop() {
	close(p.stop)
}

// Done matches the API of context.Context.Done
func (p *worker) Done() <-chan struct{} {
	return p.done
}

// Pool allows you to queue tasks for later execution
type Pool struct {
	futuresChannelMutex sync.RWMutex
	FuturesChannel      chan *Future
	stats               PoolStats
	workersMutex        sync.Mutex
	workers             []*worker
}

// PoolStats convey internal statistic information about the pool
type PoolStats struct {
	TotalItems    int64
	FailedInserts int64
}

// Close the current FuturesChannel object and stops all worker pools.  Left over items are executed synchronously
// inside Close until they are also drained.
func (p *Pool) Close() error {
	previousChan := p.closeFuturesChan()
	p.SetNumberOfWorkers(0)
	if previousChan == nil {
		return nil
	}
	// We need to drain the undrained items
	w := worker{
		stop:         make(chan struct{}),
		done:         make(chan struct{}),
		futuresQueue: previousChan,
	}
	// Start naturally ends when the passed in channel is empty && closed
	w.start()
	return nil
}

// closeFuturesChan closes the channel items are added onto, and returns that channel.  You probably want to finish
// executing undrained items.
func (p *Pool) closeFuturesChan() chan *Future {
	p.futuresChannelMutex.Lock()
	defer p.futuresChannelMutex.Unlock()
	if p.FuturesChannel != nil {
		close(p.FuturesChannel)
		ret := p.FuturesChannel
		p.FuturesChannel = nil
		return ret
	}
	return nil
}

// Stats returns internal pool tracking stats
func (p *Pool) Stats() PoolStats {
	if p == nil {
		return PoolStats{}
	}
	return PoolStats{
		TotalItems:    atomic.LoadInt64(&p.stats.TotalItems),
		FailedInserts: atomic.LoadInt64(&p.stats.FailedInserts),
	}
}

// Var exposes pool information for expvar
func (p *Pool) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		if p == nil {
			return "<nil>"
		}
		p.workersMutex.Lock()
		defer p.workersMutex.Unlock()
		p.futuresChannelMutex.RLock()
		defer p.futuresChannelMutex.RUnlock()
		return map[string]interface{}{
			"num_workers": len(p.workers),
			"chan_len":    len(p.FuturesChannel),
			"chan_cap":    cap(p.FuturesChannel),
			"stats":       p.Stats(),
		}
	})
}

// SetNumberOfWorkers increases, or decreases, the number of worker goroutines
func (p *Pool) SetNumberOfWorkers(newWorkerCount int) {
	p.workersMutex.Lock()
	defer p.workersMutex.Unlock()
	if len(p.workers) == newWorkerCount {
		return
	}
	p.futuresChannelMutex.RLock()
	defer p.futuresChannelMutex.RUnlock()
	for len(p.workers) < newWorkerCount {
		newWorker := &worker{
			stop:         make(chan struct{}),
			done:         make(chan struct{}),
			futuresQueue: p.FuturesChannel,
		}
		go newWorker.start()
		p.workers = append(p.workers, newWorker)
	}
	removedWorkers := make([]*worker, 0, len(p.workers))
	for len(p.workers) > newWorkerCount {
		lastWorker := p.workers[len(p.workers)-1]
		lastWorker.Stop()
		removedWorkers = append(removedWorkers, lastWorker)
		p.workers = p.workers[0 : len(p.workers)-1]
	}
	for _, removedWorker := range removedWorkers {
		// Wait for spawned workers to finish
		<-removedWorker.Done()
	}
}

// Offer is similar to Java's BlockingQueue offer. Offers a callback to the pool and returns a future for the callback.
// If the channel is full, will return nil.
func (p *Pool) Offer(callback Runnable) *Future {
	p.futuresChannelMutex.RLock()
	defer p.futuresChannelMutex.RUnlock()
	if p.FuturesChannel == nil {
		atomic.AddInt64(&p.stats.FailedInserts, 1)
		// Nowhere to put the future.  Returns nil
		return nil
	}
	f := NewFuture(callback)
	select {
	case p.FuturesChannel <- f:
		atomic.AddInt64(&p.stats.TotalItems, 1)
		return f
	default:
		atomic.AddInt64(&p.stats.FailedInserts, 1)
		return nil
	}
}

// Put a callback on the Pool's executor.  If the channel is full, it will block until ctx is finished, before returning
// a nil Future.  It is similar to Java's BlockingQueue Put.
func (p *Pool) Put(ctx context.Context, callback Runnable) *Future {
	p.futuresChannelMutex.RLock()
	defer p.futuresChannelMutex.RUnlock()
	if p.FuturesChannel == nil {
		atomic.AddInt64(&p.stats.FailedInserts, 1)
		return nil
	}
	f := NewFuture(callback)
	select {
	case <-ctx.Done():
		atomic.AddInt64(&p.stats.FailedInserts, 1)
		return nil
	case p.FuturesChannel <- f:
		atomic.AddInt64(&p.stats.TotalItems, 1)
		return f
	}
}

// String returns information about the pool and the pool's capacity
func (p *Pool) String() string {
	if p == nil {
		return "<nil pool>"
	}
	p.workersMutex.Lock()
	p.futuresChannelMutex.RLock()
	defer p.workersMutex.Unlock()
	defer p.futuresChannelMutex.RUnlock()
	return fmt.Sprintf("<pool workers=%d capacity=%d len=%d>", len(p.workers), len(p.FuturesChannel), cap(p.FuturesChannel))
}

// OfferOrDo is a helper around Offer that always returns a non nil Future that will eventually be executed, either
// inline (if the pool is off) or asynchronous (if the pool is on and has room).  If you want to make sure your
// callback is executed, this is probably the behavior you want.
func OfferOrDo(p *Pool, callback Runnable) *Future {
	f := p.Offer(callback)
	if f != nil {
		return f
	}
	f = NewFuture(callback)
	f.execute()
	return f
}
