package manager

import (
	"errors"
	"fmt"

	"github.com/aws/aws-sdk-go/aws/awserr"
)

var (
	// ErrNegativeLimit is returned when limit is negative
	ErrNegativeLimit = errors.New("limit must be greater than 0")

	// ErrSecretTombstoned will be returned if Tombstone is set to true when trying
	// to unmarshall a secret
	ErrSecretTombstoned = errors.New("Secret has been tombstoned")

	// ErrorSecretVersionDoesNotExist will be raised user is trying to operate with
	// version that does not exist. See Revert
	ErrorSecretVersionDoesNotExist = errors.New("manager: secret version does not exist")

	// ErrAutogenerateAndPlaintextBothSent will be raised if both autogenerated and plaintext secrets are sent
	ErrAutogenerateAndPlaintextBothSent = errors.New("both autogenerate and plaintext cannot be sent")

	// ErrInvalidClass ...
	ErrInvalidClass = errors.New("class must be in the range [1, 4]")

	// ErrSecretDoesNotExist ...
	ErrSecretDoesNotExist = errors.New("secret does not exist")

	// ErrFailedToUpdateAuditTable ...
	ErrFailedToUpdateAuditTable = errors.New("failed to update version table")
)

//InvalidRequestError represents a malformed request
type InvalidRequestError struct {
	Message error
}

// Error returns the error string
func (ire InvalidRequestError) Error() string {
	return ire.Message.Error()
}

// Code is the status code associated with the error
func (ire InvalidRequestError) Code() int { return 422 } //Unprocessable Entity

func parseGetError(secretName string, err error) error {
	if aerr, ok := err.(awserr.Error); ok {
		switch aerr.Code() {
		case "AccessDeniedException":
			return fmt.Errorf(
				"Failed to fetch secret %s. Error was: The role you specified cannot fetch %s",
				secretName,
				secretName)
		}
	}
	return fmt.Errorf("Failed to fetch secret %s. Error was: %s", secretName, err.Error())
}
