package v2

import (
	"code.justin.tv/web/cohesion/api/v2/internal/utils"
	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/rpc"

	"golang.org/x/net/context"
	"google.golang.org/grpc/codes"
)

// Associations is an operation that returns a response with all associations between 2 entities.
func (c *CohesionServer) Associations(ctx context.Context, in *rpc.RequestV2) (*rpc.GetAllResponseV2, error) {
	out := new(rpc.GetAllResponseV2)

	converted := make([]associations.Association, len(in.Associations))

	for index, assoc := range in.Associations {
		var err error

		converted[index].E1, err = utils.MakeCohesionEntity(assoc.From)
		if err != nil {
			utils.LogError(c, in.Source, err)
			return nil, err
		}

		converted[index].E2, err = utils.MakeCohesionEntity(assoc.To)
		if err != nil {
			utils.LogError(c, in.Source, err)
			return nil, err
		}
	}

	var timer *utils.Timer

	if len(converted) == 1 {
		timer = utils.NewTimer(c, "v2.get_all", in.Source)
	} else {
		timer = utils.NewTimer(c, "v2.get_all_batch", in.Source)
	}

	err := timer.Start(ctx, in.Options)
	defer timer.End()
	if err != nil {
		timer.Err = err
		utils.LogError(c, in.Source, err)
		return nil, err
	}

	var getAllResp []*associations.AssocResponseWithMeta

	//We need to validate the input for GetAllAssocBatch
	//the batch should have no more than 100 targets, and the E1 kind/id & E2 kind should match [0]
	if len(converted) < 1 {
		return nil, utils.ErrInvalidBody{Message: "Associations call requires at least one entity pair"}
	}

	if len(converted) > 100 {
		return nil, utils.ErrInvalidBody{Message: "Associations call cannot exceed 100 entity pairs"}
	}

	if len(converted) > 1 {
		for _, assoc := range converted[1:] {
			if !converted[0].E1.Equals(&assoc.E1) || converted[0].E2.Kind.String() != assoc.E2.Kind.String() {
				return nil, utils.ErrInvalidBody{Message: "When making an Associations call with multiple entity pairs, E1 must be the same for each pair and E2 must have the same Kind for each pair."}
			}
		}
	}

	if len(converted) == 1 {
		getAllResp, err = c.backend.GetAllAssoc(timer.Ctx, converted[0].E1, converted[0].E2)
	} else {
		getAllResp, err = c.backend.GetAllAssocBatch(timer.Ctx, converted...)
	}

	if err != nil {
		if _, ok := err.(associations.ErrNotFound); ok {
			return nil, errf(codes.NotFound, "%s", err.Error())
		}
		timer.Err = err
		utils.LogError(c, in.Source, err)
		return nil, err
	}

	out.From = in.Associations[0].From
	response, err := utils.MakeRPCAssocResponseWithMetas(getAllResp)
	if err != nil {
		timer.Err = err
		utils.LogError(c, in.Source, err)
		return nil, err
	}
	out.To = response

	return out, nil
}
