package v2

import (
	"log"
	"net"

	"code.justin.tv/web/cohesion/backend"
	"code.justin.tv/web/cohesion/clients"
	"code.justin.tv/web/cohesion/rpc"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/cohesion/hitcounter"
	"errors"
	"golang.org/x/net/context"
	"google.golang.org/grpc"
)

// Workaround to have `go vet` not complain about calling this function
// https://github.com/grpc/grpc-go/issues/90
var errf = grpc.Errorf

// SetupAPIV2Server creates a new CohesionServer and sets it up to start
// listening to the given bindAddress
func SetupAPIV2Server(bindAddress string, backend backend.Backender, stats statsd.Statter, errorLogger clients.ErrorLogger, hitcounter hitcounter.HitCounter) {
	// start listening to a port
	lis, err := net.Listen("tcp", bindAddress)
	if err != nil {
		log.Fatalf("main/main listening to port error=%q\n", err)
	}

	// start up a new gRPC server and create the cohesion server
	grpcServer := grpc.NewServer(grpc.UnaryInterceptor(CohesionUnaryInterceptor))
	cohesionServer := NewCohesionServer(
		backend,
		stats,
		errorLogger,
		hitcounter,
	)

	// register cohesionServer as a V2Server
	rpc.RegisterV2Server(grpcServer, cohesionServer)
	go func(server *grpc.Server) {
		log.Fatalf("main/main grpcServer Serve err = %q",
			server.Serve(lis))
	}(grpcServer)
}

// CohesionServer provides functionality required for the defined operations
// Backend implements the backing store
// Stats implements our stats logging
// ErrorLogger implements our error logging
type CohesionServer struct {
	backend     backend.Backender
	stats       statsd.Statter
	errorLogger clients.ErrorLogger
	hitcounter  hitcounter.HitCounter
}

// NewCohesionServer creates a new CohesionServer from the params
func NewCohesionServer(b backend.Backender, s statsd.Statter, e clients.ErrorLogger, h hitcounter.HitCounter) *CohesionServer {
	return &CohesionServer{
		backend:     b,
		stats:       s,
		errorLogger: e,
		hitcounter:  h,
	}
}

func CohesionUnaryInterceptor(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {

	cohesionServer, _ := info.Server.(*CohesionServer)

	genericRequest := req.(rpc.GenericRequest)
	source := genericRequest.GetSource()

	hc := cohesionServer.HitCounter()
	if hc.HasQuota(source) {
		_ = hc.Hit(source)
		return handler(ctx, req)
	}

	return nil, errors.New("Quota Exceeded")

}

// Backend returns backend param from the CohesionServer
func (C CohesionServer) Backend() backend.Backender {
	return C.backend
}

// Stats returns the stats param from the CohesionServer
func (C CohesionServer) Stats() statsd.Statter {
	return C.stats
}

// ErrorLogger returns the errorLogger from the CohesionServer
func (C CohesionServer) ErrorLogger() clients.ErrorLogger {
	return C.errorLogger
}

// HitcCunter returns the hitcounter from the CohesionServer
func (C CohesionServer) HitCounter() hitcounter.HitCounter {
	return C.hitcounter
}
