package v2

import (
	"code.justin.tv/web/cohesion/api/v2/internal/utils"
	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/backend"
	"code.justin.tv/web/cohesion/rpc"
	"golang.org/x/net/context"
	"google.golang.org/grpc/codes"
)

// Get is an operation that returns a list of responses depending on the
// associations requested
func (c *CohesionServer) Get(ctx context.Context, in *rpc.RequestV2) (*rpc.ListResponseV2, error) {
	out := new(rpc.ListResponseV2)
	assoc, err := utils.MakeCohesionAssociation(in.Associations[0])
	if err != nil {
		utils.LogError(c, in.Source, err)
		return nil, err
	}

	var timer *utils.Timer
	if assoc.E2.ID == "" {
		timer = utils.NewTimer(c, "v2.bulk_get", in.Source)
	} else {
		timer = utils.NewTimer(c, "v2.get", in.Source)
	}
	err = timer.Start(ctx, in.Options)
	defer timer.End()
	if err != nil {
		timer.Err = err
		utils.LogError(c, in.Source, err)
		return nil, err
	}

	var params *backend.Params
	if assoc.E2.ID == "" {
		params, err = utils.MakeBackendParams(in.Options)
		if err != nil {
			timer.Err = err
			utils.LogError(c, in.Source, err)
			return nil, err
		}

		params.ToKind = assoc.E2.Kind
	}

	cells, err := c.backend.GetAssoc(timer.Ctx, assoc, params)
	if err != nil {
		if _, ok := err.(associations.ErrNotFound); ok {
			return nil, errf(codes.NotFound, "%s", err.Error())
		}
		timer.Err = err
		utils.LogError(c, in.Source, err)
		return nil, err
	}

	assocs, err := utils.MakeRPCAssocResponses(cells)
	if err != nil {
		timer.Err = err
		utils.LogError(c, in.Source, err)
		return nil, err

	}

	out.From = in.Associations[0].From
	out.Kind = in.Associations[0].Kind
	out.To = assocs
	if len(cells) > 0 {
		out.Cursor = cells[len(cells)-1].Cursor
	}

	return out, nil
}
