package utils

import (
	"fmt"

	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/internal/common"
)

// ErrInvalidParam is an API-level error for invalid URL parameters
type ErrInvalidParam struct {
	Message string
}

func (E ErrInvalidParam) Error() string {
	return E.Message
}

// ErrInvalidBody is an API-level error for invalid JSON bodies
type ErrInvalidBody struct {
	Message string
}

func (E ErrInvalidBody) Error() string {
	return E.Message
}

// ErrNoCapacity is an error to indicate the pools are full
type ErrNoCapacity struct {
	Message string
}

func (E ErrNoCapacity) Error() string {
	return E.Message
}

// LogError checks the error and assigns an error code accordingly. Depending
// on the error, it will increment the stat and log the error
func LogError(T API, repo string, err error) {
	errCode := common.ErrorResponse(err)
	// specifically do not log associations.ErrNotFound - 404's are not errors.
	if _, notFoundError := err.(associations.ErrNotFound); !notFoundError && T.Stats() != nil {
		_ = T.Stats().Inc(makeErrorMetric(err, repo), 1, statsdSampleRate)
	}
	if T.ErrorLogger() != nil && common.ShouldReport(errCode) {
		T.ErrorLogger().Error(err)
	}
}

const errorMetricTemplate = "error.%v.%v"

func makeErrorMetric(err error, repo string) string {
	return fmt.Sprintf(errorMetricTemplate, repo, extractRepoFromSvcname(extractSvcnameError(err)))
}
