package utils

import (
	"encoding/json"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/backend"
	"code.justin.tv/web/cohesion/rpc"
)

// MakeBackendParams takes in an `rpc` RequestOptions and creates a
// BackendParams struct to be passed into `backend`
func MakeBackendParams(r *rpc.RequestOptions) (*backend.Params, error) {
	if r == nil {
		return nil, fmt.Errorf("No request options were specified")
	}

	params := backend.Parameters()
	if err := params.SortByString(r.Sort.String()); err != nil {
		return nil, err
	}
	params.OffsetString(strconv.FormatInt(r.Offset, 10))
	params.LimitString(strconv.FormatInt(r.Limit, 10))
	params.Cursor = r.Cursor

	return params, nil
}

// MakeRPCEntity converts an `associations` Entity into an `rpc` Entity.
// Exits early if an `associations` Entity is empty.
func MakeRPCEntity(entity *associations.Entity) (*rpc.Entity, error) {
	if entity.Empty() {
		return nil, nil
	}

	return &rpc.Entity{
		Name: entity.ID,
		Kind: entity.Kind.String(),
	}, nil
}

// MakeRPCTimestamp takes `time` Time and converts it to an `rpc` Timestamp
func MakeRPCTimestamp(t time.Time) *rpc.Timestamp {
	return &rpc.Timestamp{
		Nanos: t.UnixNano(),
	}
}

// MakeRPCAssocResponses takes in a list of `associations` AssocResponses
// and returns a list of `rpc` AssocResponses. Exits early if there is an error
func MakeRPCAssocResponses(resps []*associations.AssocResponse) ([]*rpc.AssocResponse, error) {
	responses := []*rpc.AssocResponse{}
	for _, r := range resps {
		resp, err := MakeRPCAssocResponse(r)
		if err != nil {
			return nil, err
		}
		responses = append(responses, resp)
	}
	return responses, nil
}

// MakeRPCAssocResponse converts an `associations` AssocResponse into an `rpc`
// AssocResponse
func MakeRPCAssocResponse(resp *associations.AssocResponse) (*rpc.AssocResponse, error) {
	timestamp := MakeRPCTimestamp(resp.T)
	entity, err := MakeRPCEntity(resp.E)
	if err != nil {
		return nil, err
	}

	data, err := json.Marshal(resp.D)
	if err != nil {
		return nil, err
	}

	return &rpc.AssocResponse{
		Target: entity,
		Time:   timestamp,
		Data:   data,
		Cursor: resp.Cursor,
	}, nil
}

// MakeRPCAssocResponseWithMetas converts a list of `associations` AssocResponseWithMeta into
// a list of `rpc` AssociResponseWithMeta. Exits early if there is an error.
func MakeRPCAssocResponseWithMetas(resps []*associations.AssocResponseWithMeta) ([]*rpc.AssocResponseWithMeta, error) {
	responses := []*rpc.AssocResponseWithMeta{}
	for _, r := range resps {
		resp, err := MakeRPCAssocResponseWithMeta(r)
		if err != nil {
			return nil, err
		}
		responses = append(responses, resp)
	}
	return responses, nil
}

// MakeRPCAssocResponseWithMeta converts an `associations` AssocResponseWithMeta into
// an `rpc` AssociResponseWithMeta
func MakeRPCAssocResponseWithMeta(resp *associations.AssocResponseWithMeta) (*rpc.AssocResponseWithMeta, error) {
	target, err := MakeRPCAssocResponse(resp.A)
	if err != nil {
		return nil, err
	}
	return &rpc.AssocResponseWithMeta{
		Kind:   resp.Kind,
		Target: target,
	}, nil
}

// MakeCohesionEntity converts an `rpc` Entity to an `associations` Entity
func MakeCohesionEntity(entity *rpc.Entity) (associations.Entity, error) {
	if entity == nil {
		return associations.SchemaManager.EmptyEntity, nil
	}

	kind, ok := associations.SchemaManager.EntityKind(strings.ToLower(entity.Kind))
	if !ok {
		return associations.SchemaManager.EmptyEntity, fmt.Errorf("Failed to create entity kind from string=`%v`", strings.ToLower(entity.Kind))
	}

	return associations.Entity{
		ID:   entity.Name,
		Kind: kind,
	}, nil
}

// MakeCohesionAssociations converts a list of `rpc` Associations to a list of
// `associations` Associations. Exit early when there is an error
func MakeCohesionAssociations(assocs []*rpc.Association) ([]associations.Association, error) {
	associations := []associations.Association{}
	for _, a := range assocs {
		assoc, err := MakeCohesionAssociation(a)
		if err != nil {
			return nil, err
		}
		associations = append(associations, assoc)
	}

	return associations, nil
}

// MakeCohesionAssociation converts an `rpc` Association to an `associations`
// Association. Makes sure to check BulkKind if E2 is empty (suggesting a bulk
// operation)
func MakeCohesionAssociation(assoc *rpc.Association) (associations.Association, error) {
	e1, err := MakeCohesionEntity(assoc.From)
	if err != nil {
		return associations.Association{}, fmt.Errorf("Entity error: %v", err)
	}

	e2, err := MakeCohesionEntity(assoc.To)
	if err != nil {
		return associations.Association{}, fmt.Errorf("Entity error: %v", err)
	}

	association, err := associations.NewAssoc(e1, e2, strings.ToLower(assoc.Kind))
	if err != nil {
		return association, fmt.Errorf("Assoc error: %v", err)
	}

	association.D, err = extractData(assoc.Data)
	if err != nil {
		return association, fmt.Errorf("Failed to extract data: %v", err)
	}

	if e2.Empty() {
		var ok bool
		association.E2.Kind, ok = associations.SchemaManager.EntityKind(strings.ToLower(assoc.BulkKind))
		if !ok {
			return association, fmt.Errorf("Failed to parse the bulk kind from string=`%v`", strings.ToLower(assoc.BulkKind))
		}
	}

	return association, nil
}
