package utils

import (
	"fmt"
	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/timing"
	"code.justin.tv/web/cohesion/rpc"
)

// Timer is used for stats counters and timings for gRPC operations
type Timer struct {
	// Err should only be set if an operation exits early due to an error
	Err error
	// Ctx is the timing context to be used by the operation
	Ctx context.Context

	api  API
	name string
	repo string
	xact *timing.Xact
}

// NewTimer sets up the timing transaction struct and populates the parameters
// in the Timer struct
func NewTimer(api API, name string, repo string) *Timer {
	t := new(Timer)
	t.api = api
	t.name = name
	t.repo = repo
	t.xact = &timing.Xact{
		Stats: api.Stats(),
	}
	t.xact.AddName("endpoint." + name)

	return t
}

// Start takes a context to add the timing to it
func (t *Timer) Start(ctx context.Context, opts *rpc.RequestOptions) error {
	t.xact.Start()
	t.Ctx = timing.XactContext(ctx, t.xact)
	if t.api.Stats() != nil {
		_ = t.api.Stats().Inc(fmt.Sprintf("endpoint.%v.sources.%v", t.name, t.repo), 1, statsdSampleRate)
	}

	return nil
}

// End should be called whenever an operation ends (either finishes or errors
// out). It will end the Xact timer and set the stat timing as well as log any
// panics
func (t *Timer) End() {
	if t.Err != nil {
		t.xact.End("err")
	} else {
		t.xact.End("ok")
	}

	//Don't write panics to rollbar if this is a noop stat (created by secondary reader/writer)
	_, isNoopClient := t.xact.Stats.(*statsd.NoopClient)

	if !isNoopClient {
		if p := recover(); p != nil {
			if t.api.ErrorLogger() != nil {
				t.api.ErrorLogger().Panic(p)
			}
			panic(p)
		}
	}
}
