package associations

import (
	"fmt"
	"regexp"
	"strconv"
)

// Entity represents nodes in the graph
type Entity struct {
	ID   string     `json:"id"`
	Kind EntityKind `json:"kind"`
}

var (
	rxAlpha        = regexp.MustCompile("^[a-zA-Z]+$")
	rxAlphanumeric = regexp.MustCompile("^[a-zA-Z0-9]+$")
	rxUUID         = regexp.MustCompile("^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$")
)

const (
	IDFormatInt  = "int"
	IDFormatUUID = "uuid"
)

// NewEntity makes a new Entity out of the given id and kind. Returns an error if the kind is invalid
func NewEntity(id, kind string) (Entity, error) {
	if kind, ok := SchemaManager.EntityKind(kind); ok {
		return Entity{ID: id, Kind: kind}, nil
	}
	return SchemaManager.EmptyEntity, ErrInvalidKind{fmt.Sprintf("Invalid Entity kind: %s", kind)}
}

// Validate performs basic sanity checks on the entity ID and kind
func (e Entity) Validate() error {
	// EmptyEntity is still a valid entity
	if e.Empty() {
		return nil
	}

	if len(e.ID) == 0 {
		return ErrInvalidID{"Missing ID"}
	}

	if _, ok := SchemaManager.EntityKind(e.Kind.String()); !ok {
		return ErrInvalidKind{fmt.Sprintf("Invalid entity kind %s", e.Kind.String())}
	}

	switch e.Kind.IDFormat {
	case IDFormatInt:
		if _, err := strconv.Atoi(e.ID); err != nil {
			return ErrInvalidID{fmt.Sprintf("Invalid ID %q (must be integer)", e.ID)}
		}
	case IDFormatUUID:
		if !rxUUID.MatchString(e.ID) {
			return ErrInvalidID{fmt.Sprintf("Invalid ID %q (must match regex \"%s\")", e.ID, rxUUID.String())}
		}
	default:
		return ErrInvalidKind{fmt.Sprintf("Invalid entity kind id format %s", e.Kind.IDFormat)}
	}

	return nil
}

// Empty returns true if the entity in question matches EmptyEntity
func (e Entity) Empty() bool {
	return e == SchemaManager.EmptyEntity
}

// Equals returns true if e and in are the same entity
func (e *Entity) Equals(in Comparable) bool {
	if b, ok := in.(*Entity); ok {
		if e.Kind == b.Kind && e.ID == b.ID {
			return true
		}
	}
	return false
}

func (e Entity) String() string {
	return fmt.Sprintf("{%s %s}", e.ID, e.Kind.String())
}
