package backend

import (
	"golang.org/x/net/context"
	"time"

	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/datastore"
	"github.com/cactus/go-statsd-client/statsd"
)

const statsDSampleRate = 0.1
const maxAssocsBatch = 100

// Backender details what our API layer expects to be able to call in
// the data layer
type Backender interface {
	CreateAssoc(context.Context, associations.Association) error
	DeleteAssoc(context.Context, associations.Association) error
	GetAssoc(context.Context, associations.Association, *Params) ([]*associations.AssocResponse, error)
	CountAssoc(context.Context, associations.Entity, associations.AssocKind, associations.EntityKind) (int, error)
	UpdateAssoc(context.Context, associations.Association, *UpdateParams) error
	BatchUpdateAssoc(context.Context, ...associations.Association) (int64, error)
	BatchAssoc(context.Context, ...Operation) error
	GetAllAssoc(context.Context, associations.Entity, associations.Entity) ([]*associations.AssocResponseWithMeta, error)
	GetAllAssocBatch(context.Context, ...associations.Association) ([]*associations.AssocResponseWithMeta, error)
	GetHitCounts(context.Context, string, int) (map[string]int, error)
	WriterHealth() datastore.HealthReporter
	ReaderHealth() datastore.HealthReporter
}

// Backend provides storage and stats implementations
type Backend struct {
	Writer datastore.HealthReporterWriter
	Reader datastore.HealthReporterReader

	stats statsd.Statter
}

// New returns a *Backend with minimum configuration
func New(stats statsd.Statter) *Backend {
	return &Backend{
		stats: stats,
	}
}

// WriterHealth returns the a HealthReporter for the Writer
func (b *Backend) WriterHealth() datastore.HealthReporter {
	return b.Writer
}

// ReaderHealth returns the a HealthReporter for the Reader
func (b *Backend) ReaderHealth() datastore.HealthReporter {
	return b.Reader
}

// CreateAssoc calls handlers to Create the association. For more details
// see the data source implementations at datastore/<foo>
func (b *Backend) CreateAssoc(ctx context.Context, assoc associations.Association) error {
	return b.Writer.CreateAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2, assoc.D, time.Now())
}

// DeleteAssoc calls handlers to Delete the association. For more details
// see the data source implementations at datastore/<foo>
func (b *Backend) DeleteAssoc(ctx context.Context, assoc associations.Association) error {
	if len(assoc.E2.ID) == 0 {
		return b.Writer.BulkDeleteAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2.Kind)
	}

	return b.Writer.DeleteAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2)
}

// GetAssoc calls handlers to Get the association. For more details
// see the data source implementations at datastore/<foo>
func (b *Backend) GetAssoc(ctx context.Context, assoc associations.Association, p *Params) ([]*associations.AssocResponse, error) {
	if len(assoc.E2.ID) == 0 {
		return b.Reader.BulkGetAssoc(ctx, assoc.E1, assoc.Kind, p.ToKind, p.SortBy, p.Offset, p.Limit, p.Cursor)
	}

	return b.Reader.GetAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2)
}

// CountAssoc calls handlers to Count the associations. For more details
// see the data source implementations at datastore/<foo>
func (b *Backend) CountAssoc(ctx context.Context, e1 associations.Entity, aKind associations.AssocKind, toKind associations.EntityKind) (int, error) {
	return b.Reader.CountAssoc(ctx, e1, aKind, toKind)
}

// UpdateAssoc calls handlers to Update the association. For more details
// see the data source implementations at datastore/<foo>
func (b *Backend) UpdateAssoc(ctx context.Context, assoc associations.Association, params *UpdateParams) error {
	if assoc.E2.ID == "" {
		return b.Writer.BulkUpdateAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2.Kind, params.DataBag, params.NewKind)
	}

	return b.Writer.UpdateAssoc(ctx, assoc.E1, assoc.Kind, assoc.E2, params.DataBag, params.NewKind)
}

// BatchUpdateAssoc calls handlers to Batch Update the associations. For more
// details see the data source implementations at datastore/<foo>
func (b *Backend) BatchUpdateAssoc(ctx context.Context, assocs ...associations.Association) (int64, error) {
	return b.Writer.BatchUpdateAssoc(ctx, assocs)
}

// BatchAssoc calls the other backend Assoc functions to execute the correct
// operation. Assumes the operations are all legal (checked in `api` package)
func (b *Backend) BatchAssoc(ctx context.Context, operations ...Operation) error {
	var err error
	for _, o := range operations {
		switch o.Operation {
		case "create":
			err = b.CreateAssoc(ctx, o.Assoc)
		case "delete", "bulk_delete":
			err = b.DeleteAssoc(ctx, o.Assoc)
		case "update", "bulk_update":
			err = b.UpdateAssoc(ctx, o.Assoc, o.Params)
		}

		if err != nil {
			return ErrFailedOperation{Operation: o.Operation, Message: err.Error()}
		}
	}

	return nil
}

// GetAllAssoc calls handlers to fetch the relevent associations between 2 entities. For more details
// see the data source implementations at datastore/<foo>
func (b *Backend) GetAllAssoc(ctx context.Context, e1 associations.Entity, e2 associations.Entity) ([]*associations.AssocResponseWithMeta, error) {
	return b.Reader.GetAllAssoc(ctx, e1, e2)
}

// GetAllAssocBatch calls handlers to fetch the associations between 1 entity and up to 100 others.
func (b *Backend) GetAllAssocBatch(ctx context.Context, assocs ...associations.Association) ([]*associations.AssocResponseWithMeta, error) {
	return b.Reader.GetAllAssocBatch(ctx, assocs)
}

func (b *Backend) GetHitCounts(ctx context.Context, client string, minutes int) (map[string]int, error) {
	return b.Reader.GetHitCounts(ctx, client, minutes)
}

// Health calls handlers to check the health of the datastore
// see the data source implementations at datastore/<foo>
func (b *Backend) Health(ctx context.Context) error {
	if err := b.Reader.Health(ctx); err != nil {
		return err
	}
	if err := b.Writer.Health(ctx); err != nil {
		return err
	}
	return nil
}
