package backend

import (
	"fmt"
	"strconv"

	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/datastore"
)

// UpdateParams stores the data bag and new association type when updating
// an association
type UpdateParams struct {
	DataBag map[string]interface{}
	NewKind associations.AssocKind
}

// NewUpdateParams creates an UpdateParams struct with the specified values
func NewUpdateParams(dataBag map[string]interface{}, newKind associations.AssocKind) *UpdateParams {
	return &UpdateParams{
		DataBag: dataBag,
		NewKind: newKind,
	}
}

// DefaultUpdateParams creates an UpdateParams struct with an empty map and an
// unknown association kind
func DefaultUpdateParams() *UpdateParams {
	return &UpdateParams{
		DataBag: map[string]interface{}{},
		NewKind: associations.SchemaManager.UnknownAssocKind,
	}
}

// Params specifies the valid parameters for interacting with data
// stores, affecting sorts, and page sizes
type Params struct {
	ToKind associations.EntityKind
	SortBy datastore.SortBy
	Limit  int
	Offset int
	Cursor string
}

// Parameters creates a Params struct with the default values that the
// backender interface is OK with. Feel free to over write them if you
// want, though be aware that datastore/* can do whatever they want
// with these. You're not garaunteed the page size you think you're
// going to get :)
func Parameters() *Params {
	return &Params{
		SortBy: datastore.Asc,
		Offset: 0,
		Limit:  250,
	}
}

// SortByString sets the SortBy based on the provided string,
// coverting it appropriately. An error is returned if the parameter
// supplied is not a valid SortBy. A blank string is a noop, leaving
// in place whatever value had already been set for SortBy
func (p *Params) SortByString(s string) error {
	if s == "" {
		return nil
	}

	if sortBy, ok := datastore.SortByFromString(s); ok {
		p.SortBy = sortBy
		return nil
	}

	return fmt.Errorf("Invalid value %v for parameter 'sort'", s)
}

// OffsetString sets the offset based on the provided string,
// coverting it appropriately
func (p *Params) OffsetString(s string) {
	if i, err := strconv.Atoi(s); err == nil && i > 0 {
		p.Offset = i
	}
}

// LimitString sets the limit based on the provided string,
// coverting it appropriately
func (p *Params) LimitString(s string) {
	if i, err := strconv.Atoi(s); err == nil && i > 0 {
		p.Limit = i
	}
}
