package clients

import (
	"fmt"
	"net/http"

	"code.justin.tv/common/golibs/bininfo"

	"github.com/stvp/rollbar"
)

// ErrorLogger send errors to external tracking services
type ErrorLogger interface {
	RequestError(*http.Request, error)
	RequestPanic(*http.Request, interface{})
	Error(error)
	Panic(interface{})
}

type errorLoggerImpl struct{}

// NewErrorLogger creates a rollbar logger
func NewErrorLogger(token, env string) ErrorLogger {
	rollbar.Token = token
	rollbar.Environment = env
	rollbar.CodeVersion = bininfo.Revision()
	return &errorLoggerImpl{}
}

func (l *errorLoggerImpl) RequestError(r *http.Request, err error) {
	rollbar.RequestErrorWithStack(rollbar.ERR, r, err, rollbar.BuildStack(0))
}

func (l *errorLoggerImpl) RequestPanic(r *http.Request, p interface{}) {
	err := fmt.Errorf(fmt.Sprintf("panic: %v", p))
	rollbar.RequestErrorWithStack(rollbar.ERR, r, err, rollbar.BuildStack(7))
}

func (l *errorLoggerImpl) Error(err error) {
	rollbar.ErrorWithStack(rollbar.ERR, err, rollbar.BuildStack(0))
}

func (l *errorLoggerImpl) Panic(p interface{}) {
	err := fmt.Errorf(fmt.Sprintf("panic: %v", p))
	rollbar.ErrorWithStack(rollbar.ERR, err, rollbar.BuildStack(7))
}
