package datastore

import (
	"time"

	"code.justin.tv/web/cohesion/associations"
	"golang.org/x/net/context"
)

// Named provides functions for implementations to set and fetch
// friendly names.
type Named interface {
	SetName(string)
	Name() string
}

// HealthReporter provides hooks for datastores to report their health
type HealthReporter interface {
	Health(context.Context) error
	Utilization(context.Context) float32
}

// Writer provides definitions for the functions necessary to write,
// update or delete associations
type Writer interface {
	CreateAssoc(context.Context, associations.Entity, associations.AssocKind, associations.Entity, map[string]interface{}, time.Time) error
	DeleteAssoc(context.Context, associations.Entity, associations.AssocKind, associations.Entity) error
	BulkDeleteAssoc(context.Context, associations.Entity, associations.AssocKind, associations.EntityKind) error
	UpdateAssoc(context.Context, associations.Entity, associations.AssocKind, associations.Entity, map[string]interface{}, associations.AssocKind) error
	BatchUpdateAssoc(context.Context, []associations.Association) (int64, error)
	BulkUpdateAssoc(context.Context, associations.Entity, associations.AssocKind, associations.EntityKind, map[string]interface{}, associations.AssocKind) error
}

// NamedWriter is the union of Named and Writer
type NamedWriter interface {
	Named
	Writer
}

// HealthReporterWriter is the union of HealthReporter and Writer
// +gen migrationconsumer:"errcheck, backend[Writer]" migrationproducer:"errcheck, backend[Writer]" migrationsecondary:"errcheck, backend[Writer]"
type HealthReporterWriter interface {
	HealthReporter
	NamedWriter
}

// Reader provides definitions for the functions necssary to read associations
type Reader interface {
	GetAssoc(context.Context, associations.Entity, associations.AssocKind, associations.Entity) ([]*associations.AssocResponse, error)
	BulkGetAssoc(context.Context, associations.Entity, associations.AssocKind, associations.EntityKind, SortBy, int, int, string) ([]*associations.AssocResponse, error)
	CountAssoc(context.Context, associations.Entity, associations.AssocKind, associations.EntityKind) (int, error)
	GetAllAssoc(context.Context, associations.Entity, associations.Entity) ([]*associations.AssocResponseWithMeta, error)
	GetAllAssocBatch(context.Context, []associations.Association) ([]*associations.AssocResponseWithMeta, error)
	GetHitCounts(context.Context, string, int) (map[string]int, error)
}

// NamedReader is the union of Named and Reader
type NamedReader interface {
	Named
	Reader
}

// HealthReporterReader is the union of HealthReporter and Reader
// +gen migrationsecondary:"errcheck, backend[Reader]"
type HealthReporterReader interface {
	HealthReporter
	NamedReader
}
