package datastore

import (
	"fmt"
	"strings"

	"code.justin.tv/d8a/migration"
	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/internal/common"
)

func CreatePartitionKeyBuilder() migration.PartitionKeyBuilder {
	return &cohesionPartitionKeyBuilder{}
}

type cohesionPartitionKeyBuilder struct{}

//Partition key builds a partition key from the call being made
//With bulk operations, we shard on the one entity we have
//with single row operations, we defer to the schema to tell us which entity to shard on
//with batch operations, we shard on the first assoc as a single row operation
func (self *cohesionPartitionKeyBuilder) PartitionKey(method string, params map[string]interface{}, results []interface{}) string {
	//Iterate through the parameters & determine types- there's a few types of write commands
	// - has 2 Entities & an AssocKind: single-row operation, shard on first or second entity depending on assockind
	// - has 1 Entity: bulk operation, shard on entity
	// - has array of Associations: batch operation, shard on first entity in array

	var entity1Key, entity2Key, assocKindKey1, assocKindKey2, assocSliceKey string
	for key, value := range params {
		switch value.(type) {
		case associations.Entity:
			{
				if entity1Key == "" {
					entity1Key = key
				} else if entity2Key == "" {
					entity2Key = key
				}
				break
			}
		case associations.AssocKind:
			{
				if assocKindKey1 == "" {
					assocKindKey1 = key
				} else if assocKindKey2 == "" {
					assocKindKey2 = key
				}
				break
			}
		case []associations.Association:
			{
				if assocSliceKey == "" {
					assocSliceKey = key
				}
				break
			}
		}
	}

	assocKindKey := ""
	if assocKindKey1 != "" {
		if assocKindKey2 == "" {
			assocKindKey = assocKindKey1
		} else if strings.Contains(assocKindKey1, "new") {
			assocKindKey = assocKindKey2
		} else if strings.Contains(assocKindKey2, "new") {
			assocKindKey = assocKindKey1
		} else if strings.Compare(assocKindKey1, assocKindKey2) < 0 {
			assocKindKey = assocKindKey1
		} else {
			assocKindKey = assocKindKey2
		}
	}

	if entity1Key != "" && entity2Key != "" && assocKindKey != "" {
		//Single row operation - figure out which is from & which is to
		var fromEntity, toEntity string

		if strings.Contains(entity1Key, "from") || strings.Contains(entity2Key, "to") {
			fromEntity = entity1Key
			toEntity = entity2Key
		} else if strings.Contains(entity2Key, "from") || strings.Contains(entity1Key, "to") {
			toEntity = entity1Key
			fromEntity = entity2Key
		} else if strings.Compare(entity1Key, entity2Key) < 0 {
			fromEntity = entity1Key
			toEntity = entity2Key
		} else {
			toEntity = entity1Key
			fromEntity = entity2Key
		}

		assoc, _ := params[assocKindKey].(associations.AssocKind)
		from, _ := params[fromEntity].(associations.Entity)
		to, _ := params[toEntity].(associations.Entity)

		return getAssociationShardKey(from, to, assoc)
	}

	if entity1Key != "" {
		//Bulk operation - shard on the entity
		from, _ := params[entity1Key].(associations.Entity)
		return fmt.Sprintf("%s|%s", from.Kind.KindStr, from.ID)
	}

	if assocSliceKey != "" {
		//Batch operation - shard on first associations
		assocSlice := params[assocSliceKey].([]associations.Association)
		if len(assocSlice) < 1 {
			return "replay"
		}

		return getAssociationShardKey(assocSlice[0].E1, assocSlice[0].E2, assocSlice[0].Kind)
	}

	return "replay"
}

func CreateErrorCheck() migration.ErrorCheck {
	return &cohesionErrorCheck{}
}

type cohesionErrorCheck struct{}

func (self *cohesionErrorCheck) SeriousError(err error) bool {
	resp := common.ErrorResponse(err)
	return common.ShouldReport(resp)
}

func getAssociationShardKey(from, to associations.Entity, assocKind associations.AssocKind) string {
	if assocKind.ShardOnFromEntity() {
		return fmt.Sprintf("%s|%s", from.Kind.KindStr, from.ID)
	}

	return fmt.Sprintf("%s|%s", to.Kind.KindStr, to.ID)
}
