package datastore

import (
	"code.justin.tv/d8a/migration/comparison"

	"fmt"
	"reflect"
	"time"
)

type TimestampComparePreprocessor struct{}

func (self *TimestampComparePreprocessor) Preprocess(methodName string, resultIndex int, fixture *comparison.PreprocessFixture) (bool, error) {
	if resultIndex != 0 {
		return true, nil
	}

	if methodName == "GetAssoc" || methodName == "BulkGetAssoc" {
		return self.PreprocessBasicGet(fixture)
	} else if methodName == "GetAllAssoc" || methodName == "GetAllAssocBatch" {
		return self.PreprocessMetaGet(fixture)
	}

	return true, nil
}

func (self *TimestampComparePreprocessor) PreprocessBasicGet(fixture *comparison.PreprocessFixture) (bool, error) {
	for _, pair := range fixture.ExtractValues("*", "Cursor") {
		res, err := compareCursors(pair)
		if !res || err != nil {
			return res, err
		}
	}

	for _, pair := range fixture.ExtractValues("*", "T") {
		res, err := compareTimestamps(pair)
		if !res || err != nil {
			return res, err
		}
	}

	return true, nil
}

func (self *TimestampComparePreprocessor) PreprocessMetaGet(fixture *comparison.PreprocessFixture) (bool, error) {
	for _, pair := range fixture.ExtractValues("*", "A", "Cursor") {
		res, err := compareCursors(pair)
		if !res || err != nil {
			return res, err
		}
	}

	for _, pair := range fixture.ExtractValues("*", "A", "T") {
		res, err := compareTimestamps(pair)
		if !res || err != nil {
			return res, err
		}
	}

	return true, nil
}

func compareTimestamps(pair *comparison.ValuePair) (bool, error) {
	if (pair.Old == nil) != (pair.New == nil) {
		return false, nil
	}

	oldTimeText, ok := pair.Old.(string)
	if !ok {
		return false, fmt.Errorf("%v was not a string as expected.  Was instead %s", pair.Old, reflect.TypeOf(pair.Old).Name())
	}
	newTimeText, ok := pair.New.(string)
	if !ok {
		return false, fmt.Errorf("%v was not a string as expected.  Was instead %s", pair.New, reflect.TypeOf(pair.New).Name())
	}

	oldTime := time.Time{}
	newTime := time.Time{}

	err := oldTime.UnmarshalText([]byte(oldTimeText))
	if err != nil {
		return false, err
	}

	err = newTime.UnmarshalText([]byte(newTimeText))
	if err != nil {
		return false, err
	}

	difference := newTime.Sub(oldTime)

	if difference < (-1*time.Second) || difference > time.Second {
		return false, nil
	}

	return true, nil
}

func compareCursors(pair *comparison.ValuePair) (bool, error) {
	if (pair.Old == nil) != (pair.New == nil) {
		return false, nil
	}

	oldCursor, ok := pair.Old.(string)
	if !ok {
		return false, fmt.Errorf("%v was not a string as expected.  Was instead %s", pair.Old, reflect.TypeOf(pair.Old).Name())
	}
	newCursor, ok := pair.New.(string)
	if !ok {
		return false, fmt.Errorf("%v was not a string as expected.  Was instead %s", pair.New, reflect.TypeOf(pair.New).Name())
	}

	if (len(oldCursor) == 0) != (len(newCursor) == 0) {
		return false, nil
	}

	return true, nil
}
