package datastore

import "strings"

// SortBy is a simple type reflecting which direction results should
// be returned in
type SortBy int

// Sorts are based on creation time of the association. Asc specifies
// that the oldest is listed first, desc the newest first
const (
	Asc SortBy = iota
	Desc
)

// SortByFromString returns the SortBy which the string corresponds
// to and the boolean set to true. If one is not found the default
// of Asc is returned and the return boolean is false.
func SortByFromString(sort string) (SortBy, bool) {
	switch strings.ToLower(sort) {
	case "asc":
		return Asc, true
	case "desc":
		return Desc, true
	}
	return Asc, false
}

// QueryString returns a string corresponding to the SortBy value
// which can be inserted into SQL queries.
func (sortBy SortBy) QueryString() string {
	var sortByStr string

	switch sortBy {
	case Asc:
		sortByStr = "ASC"
	case Desc:
		sortByStr = "DESC"
	default:
		sortByStr = "DESC"
	}

	return sortByStr
}
