package testds

import (
	"fmt"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/datastore"
	"errors"
)

// constants used for tests
const (
	TestID     = 123
	TestValue  = "hello"
	TestValue2 = "world"
	TestBadID  = "lkajsd"
	TestCount  = 10
)

// Backend test imple
type Backend struct {
	healthy bool
	value   string
	time    time.Time
	name    string
}

// New test impl
func New(healthy bool, t time.Time) *Backend {
	return &Backend{
		healthy: healthy,
		value:   TestValue,
		time:    t,
	}
}

// Health test impl
func (B *Backend) Health(ctx context.Context) error {
	if !B.healthy {
		return fmt.Errorf("")
	}
	return nil
}

// CreateAssoc test impl
func (B *Backend) CreateAssoc(ctx context.Context,
	from associations.Entity, assocKind associations.AssocKind, to associations.Entity, data map[string]interface{}, creationDate time.Time) error {
	return nil
}

// UpdateAssoc test impl
func (B *Backend) UpdateAssoc(ctx context.Context,
	from associations.Entity, assocKind associations.AssocKind, to associations.Entity, data map[string]interface{}, newKind associations.AssocKind) error {
	if from.ID == "777" {
		return associations.ErrNotFound{Assoc: &associations.Association{
			E1:   from,
			E2:   to,
			Kind: assocKind,
		}}
	}
	return nil
}

// BatchUpdateAssoc test impl
func (B *Backend) BatchUpdateAssoc(ctx context.Context, assocs []associations.Association) (int64, error) {
	rowsAffected := int64(len(assocs))
	e1 := assocs[0].E1
	assocKind := assocs[0].Kind

	if e1.ID == "777" {
		return int64(0), associations.ErrNotFound{Assoc: &associations.Association{
			E1:   e1,
			E2:   associations.SchemaManager.EmptyEntity,
			Kind: assocKind,
		}}
	}

	for _, assoc := range assocs {
		e2 := assoc.E2
		if e2.ID == "888" {
			rowsAffected--
		}
	}
	return rowsAffected, nil
}

// BulkUpdateAssoc test impl
func (B *Backend) BulkUpdateAssoc(ctx context.Context,
	entity associations.Entity, assocKind associations.AssocKind, toKind associations.EntityKind, data map[string]interface{}, newKind associations.AssocKind) error {
	if entity.ID == "777" {
		return associations.ErrNotFound{Assoc: &associations.Association{
			E1:   entity,
			E2:   associations.SchemaManager.EmptyEntity,
			Kind: assocKind,
		}}
	}
	return nil
}

// DeleteAssoc test impl
func (B *Backend) DeleteAssoc(ctx context.Context,
	from associations.Entity, assocKind associations.AssocKind, to associations.Entity) (err error) {
	return nil
}

// BulkDeleteAssoc test impl
func (B *Backend) BulkDeleteAssoc(ctx context.Context, entity associations.Entity, assocKind associations.AssocKind, toKind associations.EntityKind) error {
	if entity.ID == "777" {
		return associations.ErrNotFound{}
	}
	return nil
}

// GetAssoc test impl
func (B *Backend) GetAssoc(ctx context.Context,
	from associations.Entity, assocKind associations.AssocKind, to associations.Entity) ([]*associations.AssocResponse, error) {
	if to.ID == "777" {
		return nil, associations.ErrNotFound{}
	}

	return []*associations.AssocResponse{
		&associations.AssocResponse{
			E: &to,
			T: B.time,
			D: map[string]interface{}{"foo": "bar"}},
	}, nil
}

// BulkGetAssoc test impl
func (B *Backend) BulkGetAssoc(ctx context.Context, from associations.Entity, assocKind associations.AssocKind,
	toKind associations.EntityKind, sortBy datastore.SortBy, offset, limit int, cursor string) ([]*associations.AssocResponse, error) {

	if from.ID == "777" {
		return []*associations.AssocResponse{}, nil
	}

	return []*associations.AssocResponse{
		&associations.AssocResponse{
			E: &associations.Entity{ID: "456", Kind: associations.UserKind},
			T: B.time,
			D: map[string]interface{}{"foo": "bar"},
		},
	}, nil
}

// CountAssoc test impl
func (B *Backend) CountAssoc(ctx context.Context, from associations.Entity, assocKind associations.AssocKind, toKind associations.EntityKind) (int, error) {
	return 0, nil
}

// GetAllAssoc test impl
func (B *Backend) GetAllAssoc(ctx context.Context, from associations.Entity, to associations.Entity) ([]*associations.AssocResponseWithMeta, error) {
	if from.ID == "777" {
		return []*associations.AssocResponseWithMeta{}, nil
	}

	return []*associations.AssocResponseWithMeta{
		&associations.AssocResponseWithMeta{
			Kind: "foobar",
			A: &associations.AssocResponse{
				E: &to,
				T: B.time,
				D: map[string]interface{}{"foo": "bar"},
			},
		},
	}, nil
}

// GetAllAssocBatch test impl
func (B *Backend) GetAllAssocBatch(ctx context.Context, assocs []associations.Association) ([]*associations.AssocResponseWithMeta, error) {
	if assocs[0].E1.ID == "777" {
		return []*associations.AssocResponseWithMeta{}, nil
	}

	response := []*associations.AssocResponseWithMeta{}

	for _, assoc := range assocs {
		response = append(response, &associations.AssocResponseWithMeta{
			Kind: "foobar",
			A: &associations.AssocResponse{
				E: &associations.Entity{ID: assoc.E2.ID, Kind: assoc.E2.Kind},
				T: B.time,
				D: map[string]interface{}{"foo": "bar"},
			},
		})
	}

	return response, nil
}

func (B *Backend) GetHitCounts(ctx context.Context, client string, minutes int) (map[string]int, error) {
	return nil, errors.New("GetHitCounts not yet impelemented in testds.")
}

// Utilization test impl
func (B *Backend) Utilization(ctx context.Context) float32 {
	if B.healthy {
		return 0.0
	}
	return 0.9
}

// Set supports setting test data
func (B *Backend) Set(ctx context.Context, id int, value string) error {
	if id == TestID {
		B.value = value
	}
	return nil
}

// SetName will give this backend the provided name
func (B *Backend) SetName(name string) {
	B.name = name
}

// Name will return the name of this backend
func (B *Backend) Name() string {
	return B.name
}
