package datastore

import (
	"fmt"
	"strconv"
)

// TimeStamp is a 64-bit integer that represents a specific point in time.
type TimeStamp int64

// MakeTimestampFromString parses a string into a timestamp.  Can return an error
// if there are problems with parsing.  Otherwise returns nil for the error
// val.
func MakeTimestampFromString(time string) (TimeStamp, error) {
	inttime, err := strconv.ParseInt(time, 10, 64)

	if err != nil {
		return 0, err
	}

	return TimeStamp(inttime), nil
}

// MakeTimestampFromInt casts an int64 into a timestamp.  This doesn't really do much.
func MakeTimestampFromInt(time int64) TimeStamp {
	return TimeStamp(time)
}

// QueryString parses the timestamp into a string in a format that Postgres will accept.
// We use an extremely precise float value to avoid any off by one errors when querying for
// a timestamp in postgres.  The `to_timestamp` function in postgres takes in a timestamp
// in double precision seconds.
func (time TimeStamp) QueryString() string {
	return fmt.Sprintf("%.9f", float64(time)/1000000000)
}
