package twirprpc

import (
	"time"

	"errors"

	"code.justin.tv/feeds/graphdb/proto/datastorerpc"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/web/cohesion/associations"
	"code.justin.tv/web/cohesion/datastore"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/timestamp"
	"golang.org/x/net/context"
)

type TwirpRPC struct {
	Reader datastorerpc.Reader
	Writer datastorerpc.Writer
	// Mirror, if true tees writes to cohesion's database
	Mirror   bool
	DataName string
}

var _ datastore.Reader = &TwirpRPC{}
var _ datastore.Writer = &TwirpRPC{}
var _ datastore.Named = &TwirpRPC{}
var _ datastore.HealthReporter = &TwirpRPC{}

// Writer Methods
func (t *TwirpRPC) CreateAssoc(ctx context.Context, from associations.Entity, kind associations.AssocKind, to associations.Entity, data map[string]interface{}, time time.Time) error {
	params := &datastorerpc.CreateAssocRequest{
		From:         toC2Ent(from),
		AssocKind:    toC2AssocKind(kind),
		To:           toC2Ent(to),
		Data:         toC2DataBag(data),
		CreationDate: toProtoTime(time),
	}
	if t.Mirror {
		params.Opt = &datastorerpc.WriteReqestOptions{
			Mirror: true,
		}
	}

	resp, err := t.Writer.CreateAssoc(ctx, params)
	if resp.Exists != nil {
		return associations.ErrAssociationExists{Source: errors.New(resp.GetExists().GetSource().GetMessage())}
	}
	return err
}

func (t *TwirpRPC) DeleteAssoc(ctx context.Context, from associations.Entity, kind associations.AssocKind, to associations.Entity) error {
	params := &datastorerpc.DeleteAssocRequest{
		From:      toC2Ent(from),
		AssocKind: toC2AssocKind(kind),
		To:        toC2Ent(to),
	}
	if t.Mirror {
		params.Opt = &datastorerpc.WriteReqestOptions{
			Mirror: true,
		}
	}
	resp, err := t.Writer.DeleteAssoc(ctx, params)
	if resp.NotFound != nil {
		return associations.ErrNotFound{
			Assoc: getAssoc(resp.GetNotFound().GetAssoc()),
		}
	}
	return err
}

func getAssoc(assoc *datastorerpc.AssociationV1) *associations.Association {
	return &associations.Association{
		E1:   *toC1Ent(assoc.GetE1()),
		E2:   *toC1Ent(assoc.GetE2()),
		Kind: getAssocKind(assoc.GetKind()),
		D:    toC1DataBag(assoc.GetD()),
	}
}

func getAssocKind(kind *datastorerpc.AssocKind) associations.AssocKind {
	return associations.AssocKind{
		KindStr:    kind.GetKindStr(),
		InverseStr: kind.GetInverseStr(),
		E1KindStr:  kind.GetE1KindStr(),
		E2KindStr:  kind.GetE2KindStr(),
		ShardFrom:  kind.GetShardFrom(),
	}
}

func (t *TwirpRPC) BulkDeleteAssoc(ctx context.Context, from associations.Entity, kind associations.AssocKind, toKind associations.EntityKind) error {
	params := &datastorerpc.BulkDeleteAssocRequest{
		From:      toC2Ent(from),
		AssocKind: toC2AssocKind(kind),
		ToKind:    toC2EntKind(toKind),
	}
	if t.Mirror {
		params.Opt = &datastorerpc.WriteReqestOptions{
			Mirror: true,
		}
	}
	_, err := t.Writer.BulkDeleteAssoc(ctx, params)
	return err
}

func (t *TwirpRPC) UpdateAssoc(ctx context.Context, from associations.Entity, kind associations.AssocKind, to associations.Entity, data map[string]interface{}, toKind associations.AssocKind) error {
	params := &datastorerpc.UpdateAssocRequest{
		From:      toC2Ent(from),
		AssocKind: toC2AssocKind(kind),
		To:        toC2Ent(to),
		Data:      toC2DataBag(data),
		NewKind:   toC2AssocKind(toKind),
	}
	if t.Mirror {
		params.Opt = &datastorerpc.WriteReqestOptions{
			Mirror: true,
		}
	}
	resp, err := t.Writer.UpdateAssoc(ctx, params)
	if resp.NotFound != nil {
		return associations.ErrNotFound{
			Assoc: getAssoc(resp.GetNotFound().GetAssoc()),
		}
	}
	return err
}

func (t *TwirpRPC) BatchUpdateAssoc(ctx context.Context, from []associations.Association) (int64, error) {
	assocs := []*datastorerpc.AssociationV1{}
	for _, assoc := range from {
		assocs = append(assocs, toC2Assoc(assoc))
	}
	params := &datastorerpc.BatchUpdateAssocRequest{
		Assocs: assocs,
	}
	if t.Mirror {
		params.Opt = &datastorerpc.WriteReqestOptions{
			Mirror: true,
		}
	}
	_, err := t.Writer.BatchUpdateAssoc(ctx, params)
	return 1, err
}

func (t *TwirpRPC) BulkUpdateAssoc(ctx context.Context, to associations.Entity, assocKind associations.AssocKind, toKind associations.EntityKind, data map[string]interface{}, newKind associations.AssocKind) error {
	params := &datastorerpc.BulkUpdateAssocRequest{
		Entity:    toC2Ent(to),
		AssocKind: toC2AssocKind(assocKind),
		ToKind:    toC2EntKind(toKind),
		Data:      toC2DataBag(data),
		NewKind:   toC2AssocKind(newKind),
	}
	if t.Mirror {
		params.Opt = &datastorerpc.WriteReqestOptions{
			Mirror: true,
		}
	}
	_, err := t.Writer.BulkUpdateAssoc(ctx, params)
	return err
}

// Reader methods
func (t *TwirpRPC) GetAssoc(ctx context.Context, from associations.Entity, kind associations.AssocKind, to associations.Entity) ([]*associations.AssocResponse, error) {
	params := &datastorerpc.GetAssocRequest{
		From:      toC2Ent(from),
		AssocKind: toC2AssocKind(kind),
		To:        toC2Ent(to),
	}
	resp, err := t.Reader.GetAssoc(ctx, params)
	if resp.NotFound != nil {
		return nil, associations.ErrNotFound{
			Assoc: getAssoc(resp.GetNotFound().GetAssoc()),
		}
	}
	return toC1AssocRespList(resp.Assocs), err
}

func (t *TwirpRPC) BulkGetAssoc(ctx context.Context, from associations.Entity, assocKind associations.AssocKind, entKind associations.EntityKind, sort datastore.SortBy, offset int, limit int, cursor string) ([]*associations.AssocResponse, error) {
	sortString := sort.QueryString()
	sortByValue := datastorerpc.BulkGetAssocRequest_DESC
	if sortString == "ASC" {
		sortByValue = datastorerpc.BulkGetAssocRequest_ASC
	}
	params := &datastorerpc.BulkGetAssocRequest{
		From:      toC2Ent(from),
		AssocKind: toC2AssocKind(assocKind),
		ToKind:    toC2EntKind(entKind),
		SortBy:    sortByValue,
		Offset:    int64(offset),
		Limit:     int64(limit),
		Cursor:    cursor,
	}
	resp, err := t.Reader.BulkGetAssoc(ctx, params)
	return toC1AssocRespList(resp.Assocs), err
}

func (t *TwirpRPC) CountAssoc(ctx context.Context, from associations.Entity, assocKind associations.AssocKind, entKind associations.EntityKind) (int, error) {
	params := &datastorerpc.CountAssocRequest{
		From:      toC2Ent(from),
		AssocKind: toC2AssocKind(assocKind),
		ToKind:    toC2EntKind(entKind),
	}
	resp, err := t.Reader.CountAssoc(ctx, params)
	return int(resp.Count), err
}

func (t *TwirpRPC) GetAllAssoc(ctx context.Context, from associations.Entity, to associations.Entity) ([]*associations.AssocResponseWithMeta, error) {
	params := &datastorerpc.GetAllAssocRequest{
		From: toC2Ent(from),
		To:   toC2Ent(to),
	}
	resp, err := t.Reader.GetAllAssoc(ctx, params)
	return toC1AssocRespWithMetaList(resp.Assocs), err
}

func (t *TwirpRPC) GetAllAssocBatch(ctx context.Context, from []associations.Association) ([]*associations.AssocResponseWithMeta, error) {
	assocs := []*datastorerpc.AssociationV1{}
	for _, assoc := range from {
		assocs = append(assocs, toC2Assoc(assoc))
	}
	resp, err := t.Reader.GetAllAssocBatch(ctx, &datastorerpc.GetAllAssocBatchRequest{
		Assocs: assocs,
	})
	return toC1AssocRespWithMetaList(resp.Assocs), err
}

func (t *TwirpRPC) UpdateOrInsertHitCount(ctx context.Context, client string, newHits int, bucketDuration time.Duration) error {
	return nil
}

func (t *TwirpRPC) GetHitCounts(ctx context.Context, s string, a int) (map[string]int, error) {
	return nil, nil
}

func toC1AssocRespList(assocs []*datastorerpc.AssocResponse) []*associations.AssocResponse {
	assocList := []*associations.AssocResponse{}
	for _, assoc := range assocs {
		assocList = append(assocList, toC1AssocResponse(assoc))
	}
	return assocList
}

func toC1AssocResponse(assoc *datastorerpc.AssocResponse) *associations.AssocResponse {
	t, _ := ptypes.Timestamp(assoc.T)
	return &associations.AssocResponse{
		E:      toC1Ent(assoc.E),
		T:      t,
		D:      toC1DataBag(assoc.D),
		Cursor: assoc.Cursor,
	}
}

func toC1AssocResponseMeta(assocMeta *datastorerpc.AssocResponseWithMeta) *associations.AssocResponseWithMeta {
	return &associations.AssocResponseWithMeta{
		Kind: assocMeta.Kind,
		A:    toC1AssocResponse(assocMeta.A),
	}
}

func toC1AssocRespWithMetaList(assocsMeta []*datastorerpc.AssocResponseWithMeta) []*associations.AssocResponseWithMeta {
	assocMetaList := []*associations.AssocResponseWithMeta{}
	for _, assocM := range assocsMeta {
		assocMetaList = append(assocMetaList, toC1AssocResponseMeta(assocM))
	}
	return assocMetaList
}

func toC2Assoc(assoc associations.Association) *datastorerpc.AssociationV1 {
	return &datastorerpc.AssociationV1{
		E1:   toC2Ent(assoc.E1),
		E2:   toC2Ent(assoc.E2),
		Kind: toC2AssocKind(assoc.Kind),
		D:    toC2DataBag(assoc.D),
	}
}

func toC2AssocKind(kind associations.AssocKind) *datastorerpc.AssocKind {
	return &datastorerpc.AssocKind{
		E1KindStr:  kind.E1KindStr,
		E2KindStr:  kind.E2KindStr,
		InverseStr: kind.InverseStr,
		KindStr:    kind.KindStr,
		ShardFrom:  kind.ShardFrom,
	}
}

func toC1DataBag(from *graphdb.DataBag) associations.DataBag {
	data := make(associations.DataBag)
	i, s, b, d := dataBagTypes(from)
	for k, v := range i {
		data[k] = v
	}
	for k, v := range s {
		data[k] = v
	}
	for k, v := range b {
		data[k] = v
	}
	for k, v := range d {
		data[k] = v
	}
	if len(data) == 0 {
		return nil
	}
	return data
}

// Types returns a copy of the contents of DataBag.
//
// NOTE: Types assumes caller has initialized d to non-nil value
func dataBagTypes(dataBag *graphdb.DataBag) (map[string]int64, map[string]string, map[string]bool, map[string]float64) {
	if dataBag == nil {
		return nil, nil, nil, nil
	}
	ints := make(map[string]int64, len(dataBag.Ints))
	for k, v := range dataBag.Ints {
		ints[k] = v
	}

	strs := make(map[string]string, len(dataBag.Strings))
	for k, v := range dataBag.Strings {
		strs[k] = v
	}

	doubles := make(map[string]float64, len(dataBag.Doubles))
	for k, v := range dataBag.Doubles {
		doubles[k] = v
	}

	bools := make(map[string]bool, len(dataBag.Bools))
	for k, v := range dataBag.Bools {
		bools[k] = v
	}
	return ints, strs, bools, doubles
}

func toC2DataBag(fromData map[string]interface{}) *graphdb.DataBag {
	d := &graphdb.DataBag{}
	for k, v := range fromData {
		switch casted := v.(type) {
		case float64:
			if d.Doubles == nil {
				d.Doubles = make(map[string]float64)
			}
			d.Doubles[k] = casted
		case int64:
			if d.Ints == nil {
				d.Ints = make(map[string]int64)
			}
			d.Ints[k] = casted
		case int:
			if d.Ints == nil {
				d.Ints = make(map[string]int64)
			}
			d.Ints[k] = int64(casted)
		case bool:
			if d.Bools == nil {
				d.Bools = make(map[string]bool)
			}
			d.Bools[k] = casted
		case string:
			if d.Strings == nil {
				d.Strings = make(map[string]string)
			}
			d.Strings[k] = casted
		}
	}
	return d
}

func toC1Ent(entity *datastorerpc.EntityV1) *associations.Entity {
	return &associations.Entity{
		ID:   entity.Id,
		Kind: toC1EntKind(entity.Kind),
	}
}

func toC2Ent(entity associations.Entity) *datastorerpc.EntityV1 {
	return &datastorerpc.EntityV1{
		Id:   entity.ID,
		Kind: toC2EntKind(entity.Kind),
	}
}

func toC1EntKind(kind *datastorerpc.EntityKind) associations.EntityKind {
	return associations.EntityKind{
		IDFormat: kind.IdFormat,
		KindStr:  kind.KindStr,
	}
}

func toC2EntKind(kind associations.EntityKind) *datastorerpc.EntityKind {
	return &datastorerpc.EntityKind{
		IdFormat: kind.IDFormat,
		KindStr:  kind.KindStr,
	}
}

func toProtoTime(t time.Time) *timestamp.Timestamp {
	n := t.UnixNano()
	return &timestamp.Timestamp{
		Seconds: n / time.Second.Nanoseconds(),
		Nanos:   int32(n % time.Second.Nanoseconds()),
	}
}

//Named method
func (t *TwirpRPC) SetName(name string) {
	t.DataName = name
}

func (t *TwirpRPC) Name() string {
	return t.DataName
}

//HealthReporter Methods
func (t *TwirpRPC) Health(ctx context.Context) error {
	return nil
}

func (t *TwirpRPC) Utilization(ctx context.Context) float32 {
	return float32(0)
}
