package common

import (
	"net/http"

	"github.com/jixwanwang/apiutils"

	"code.justin.tv/web/cohesion/associations"
)

// ErrInvalidParam is an API-level error for invalid URL parameters
type ErrInvalidParam struct {
	Message string
}

func (E ErrInvalidParam) Error() string {
	return E.Message
}

// ErrInvalidBody is an API-level error for invalid JSON bodies
type ErrInvalidBody struct {
	Message string
}

func (E ErrInvalidBody) Error() string {
	return E.Message
}

// ErrNoCapacity is an error to indicate the pools are full
type ErrNoCapacity struct {
	Message string
}

func (T ErrNoCapacity) Error() string {
	return T.Message
}

// ErrorResponse retrieves an ErrorResponse obj from a given error
func ErrorResponse(err error) apiutils.ErrorResponse {
	switch t := err.(type) {
	case associations.ErrAssociationExists:
		return apiutils.NewErrorResponse(apiutils.StatusUnprocessableEntity, t.Error())
	case associations.ErrInvalidID, associations.ErrInvalidKind, ErrInvalidParam, ErrInvalidBody:
		return apiutils.NewErrorResponse(http.StatusBadRequest, t.Error())
	case associations.ErrNotFound:
		return apiutils.NewErrorResponse(http.StatusNotFound, t.Error())
	case ErrNoCapacity:
		return apiutils.NewErrorResponse(http.StatusServiceUnavailable, t.Error())
	default:
		return apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error())
	}
}

// ShouldReport determines whether a given error and its error response should be considered
// the sort of critical error that should be logged & statted
func ShouldReport(errRes apiutils.ErrorResponse) bool {
	return errRes.Status >= http.StatusInternalServerError
}
