package main

import (
	"fmt"
	"io"
	"log"
	"net/http"
	"time"

	"io/ioutil"
	"os"
	"strings"
	"sync/atomic"

	"code.justin.tv/feeds/consulconf"
	"code.justin.tv/feeds/distconf"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/hashicorp/consul/api"
)

func localIP() string {
	localIPResp, err := http.Get("http://169.254.169.254/latest/meta-data/local-ipv4/")
	if err != nil {
		fmt.Printf("Unable to get local IP: %s\n", err.Error())
		return localHostname()
	}
	localIP, err := ioutil.ReadAll(localIPResp.Body)
	if err != nil {
		fmt.Printf("Unable to read body: %s\n", err.Error())
		return localHostname()
	}
	return string(localIP)
}

func sanitizeStatsd(s string) string {
	if len(s) > 32 {
		s = s[0:32]
	}
	return strings.Replace(s, ".", "_", -1)
}

func localHostname() string {
	// Testing pull request
	localIPResp, err := http.Get("http://169.254.169.254/latest/meta-data/hostname/")
	if err != nil {
		fmt.Printf("Unable to get local hostname: %s\n", err.Error())
		hostname, err2 := os.Hostname()
		if err != nil {
			fmt.Println("Unable to get hostname", err2)
			return "unknownhostname"
		}
		return hostname
	}
	localHostname, err := ioutil.ReadAll(localIPResp.Body)
	if err != nil {
		fmt.Printf("Unable to read body: %s\n", err.Error())
		hostname, err2 := os.Hostname()
		if err != nil {
			fmt.Println("Unable to get hostname", err2)
			return "unknownhostname"
		}
		return hostname
	}
	return string(localHostname)
}

func handleItem(statsdClient statsd.Statter, dconf *distconf.Distconf) http.Handler {
	tableName := dconf.Str("helloworld.items_table", "")
	if tableName.Get() == "" {
		panic("Cannot find table name")
	}
	return http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		fmt.Println("Item!")
		_ = statsdClient.Inc("req_count", 1, 1.0)
		region := &aws.Config{Region: aws.String("us-west-2")}
		session, err2 := session.NewSession(region)
		if err2 != nil {
			rw.WriteHeader(http.StatusInternalServerError)
			if _, err3 := io.WriteString(rw, err2.Error()); err3 != nil {
				fmt.Println("Unable to write string", err3)
			}
			return
		}
		dbClient := dynamodb.New(session)
		_, err2 = dbClient.PutItem(&dynamodb.PutItemInput{
			TableName: aws.String(tableName.Get()),
			Item: map[string]*dynamodb.AttributeValue{
				"id": {
					S: aws.String(time.Now().String()),
				},
			},
		})
		if err2 != nil {
			rw.WriteHeader(http.StatusInternalServerError)
			if _, err3 := io.WriteString(rw, err2.Error()); err3 != nil {
				fmt.Println("Unable to write string", err3)
			}
			return
		}
		_, err2 = io.WriteString(rw, "write ok v14!"+os.Getenv("GIT_COMMIT"))
		if err2 != nil {
			panic(err2)
		}
	})
}

func handleRunning(isHealthy *int64) http.Handler {
	return http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		if atomic.LoadInt64(isHealthy) == 0 {
			rw.WriteHeader(http.StatusNotFound)
			fmt.Println("Failed healthcheck")
			return
		}
		_, err2 := io.WriteString(rw, "running!")
		if err2 != nil {
			panic(err2)
		}
		fmt.Println("Healthcheck!")
	})
}

func setupDconf(env string) (*distconf.Distconf, error) {
	config := api.DefaultConfig()
	config.Address = "consul.internal.justin.tv"
	config.Datacenter = "us-west2"

	client, err := api.NewClient(config)
	if err != nil {
		return nil, err
	}
	kv := client.KV()
	readers := make([]distconf.Reader, 0)
	readers = append(readers, &consulconf.Consulconf{
		Prefix: "settings/feeds/" + env,
		KV:     kv,
	})
	if env == "canary" {
		readers = append(readers, &consulconf.Consulconf{
			Prefix: "settings/feeds/production",
			KV:     kv,
		})
	}

	dconf := &distconf.Distconf{
		Readers: readers,
	}
	return dconf, nil
}

func main() {
	localIP := localIP()
	myHostname := localHostname()
	fmt.Println("IP and hostname", localIP, myHostname)
	env := os.Getenv("ENVIRONMENT")
	dconf, err := setupDconf(env)
	if err != nil {
		panic(err)
	}
	statsdClient, err := statsd.NewBufferedClient(localIP+":8125", "feeds.helloworld."+env+"."+sanitizeStatsd(myHostname), time.Second, 512)
	if err != nil {
		fmt.Println("not using statsd client", err)
		statsdClient, err = statsd.NewNoopClient()
		if err != nil {
			panic(err)
		}
	}
	hostname, err := os.Hostname()
	for _, s := range os.Environ() {
		fmt.Println("Environ: ", s)
	}
	fmt.Println("Hostname is ", hostname, err)
	go func() {
		for {
			time.Sleep(time.Second)
			_ = statsdClient.Inc("is_running", 1, 1.0)
		}
	}()
	mux := http.NewServeMux()
	mux.Handle("/item", handleItem(statsdClient, dconf))
	isHealthy := int64(1)
	mux.Handle("/debug/running", handleRunning(&isHealthy))

	mux.Handle("/healthy", http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		atomic.StoreInt64(&isHealthy, 1)
		fmt.Println("Now healthy!")
	}))

	mux.Handle("/unhealthy", http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		atomic.StoreInt64(&isHealthy, 0)
		fmt.Println("unhealthhy")
	}))
	go func() {
		log.Println(http.ListenAndServe(":6060", nil))
	}()

	s := http.Server{}
	s.Addr = ":8080"
	s.Handler = mux
	fmt.Println("Listening")
	fmt.Println(s.ListenAndServe())
}
