package hystrix

import (
	"sync"
	"sync/atomic"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

func TestGetCircuit(t *testing.T) {

	Convey("when calling GetCircuit", t, func() {
		circuits := NewCircuits()
		var created bool
		var err error
		_, created, err = circuits.getCircuit("foo")

		Convey("once, the circuit should be created", func() {
			So(err, ShouldBeNil)
			So(created, ShouldEqual, true)
		})

		Convey("twice, the circuit should be reused", func() {
			_, created, err = circuits.getCircuit("foo")
			So(err, ShouldBeNil)
			So(created, ShouldEqual, false)
		})
	})
}

func TestMultithreadedGetCircuit(t *testing.T) {

	Convey("calling GetCircuit", t, func() {
		circuits := NewCircuits()

		numThreads := 100
		var numCreates int32
		var numRunningRoutines int32
		var startingLine sync.WaitGroup
		var finishLine sync.WaitGroup
		startingLine.Add(1)
		finishLine.Add(numThreads)

		for i := 0; i < numThreads; i++ {
			go func() {
				if atomic.AddInt32(&numRunningRoutines, 1) == int32(numThreads) {
					startingLine.Done()
				} else {
					startingLine.Wait()
				}

				_, created, _ := circuits.getCircuit("foo")

				if created {
					atomic.AddInt32(&numCreates, 1)
				}

				finishLine.Done()
			}()
		}

		finishLine.Wait()

		Convey("should be threadsafe", func() {
			So(numCreates, ShouldEqual, int32(1))
		})
	})
}

func TestReportEventOpenThenClose(t *testing.T) {
	Convey("when a circuit is closed", t, func() {
		circuits := NewCircuits()
		circuits.ConfigureCommand("", CommandConfig{ErrorPercentThreshold: 50})

		cb, _, err := circuits.getCircuit("")
		So(err, ShouldEqual, nil)
		So(cb.IsOpen(), ShouldBeFalse)
		openedTime := cb.openedOrLastTestedTime

		Convey("but the metrics are unhealthy", func() {
			cb.metrics = metricFailingPercent(100, newSettingsCollection())
			So(cb.metrics.IsHealthy(time.Now()), ShouldBeFalse)

			Convey("and a success is reported", func() {
				err = cb.ReportEvent([]string{"success"}, time.Now(), 0)
				So(err, ShouldEqual, nil)

				Convey("the circuit does not open then close", func() {
					So(cb.openedOrLastTestedTime, ShouldEqual, openedTime)
				})
			})
		})
	})
}
