package hystrix

import (
	"log"
	"sync"
	"time"

	"code.justin.tv/feeds/hystrix-go/hystrix/metric_collector"
)

type runFunc func() error
type fallbackFunc func(error) error
type InitMetricCollectorFunc func(name string) metricCollector.MetricCollector

// A CircuitError is an error which models various failure states of execution,
// such as the circuit being open or a timeout.
type CircuitError struct {
	Message string
}

func (e CircuitError) Error() string {
	return "hystrix: " + e.Message
}

var (
	// ErrMaxConcurrency occurs when too many of the same named command are executed at the same time.
	ErrMaxConcurrency = CircuitError{Message: "max concurrency"}
	// ErrCircuitOpen returns when an execution attempt "short circuits". This happens due to the circuit being measured as unhealthy.
	ErrCircuitOpen = CircuitError{Message: "circuit open"}
	// ErrTimeout occurs when the provided function takes too long to execute.
	ErrTimeout = CircuitError{Message: "timeout"}

	defaultCircuits *Circuits
)

const (
	// DefaultTimeout is how long to wait for command to complete, in milliseconds
	DefaultTimeout = 1000
	// DefaultMaxConcurrent is how many commands of the same type can run at the same time
	DefaultMaxConcurrent = 10
	// DefaultVolumeThreshold is the minimum number of requests needed before a circuit can be tripped due to health
	DefaultVolumeThreshold = 20
	// DefaultSleepWindow is how long, in milliseconds, to wait after a circuit opens before testing for recovery
	DefaultSleepWindow = 5000
	// DefaultErrorPercentThreshold causes circuits to open once the rolling measure of errors exceeds this percent of requests
	DefaultErrorPercentThreshold = 50
)

// CommandConfig is used to tune circuit settings at runtime
type CommandConfig struct {
	Timeout                int `json:"timeout"`
	MaxConcurrentRequests  int `json:"max_concurrent_requests"`
	RequestVolumeThreshold int `json:"request_volume_threshold"`
	SleepWindow            int `json:"sleep_window"`
	ErrorPercentThreshold  int `json:"error_percent_threshold"`
}

type Circuits struct {
	Breakers                 map[string]*CircuitBreaker
	BreakersMutex            *sync.RWMutex
	Settings                 *settingsCollection
	InitMetricCollectorFuncs []InitMetricCollectorFunc
}

func NewCircuits() *Circuits {
	return NewCircuitsWithMetrics(nil)
}

func NewCircuitsWithMetrics(initMetricCollectorFuncs []InitMetricCollectorFunc) *Circuits {
	return &Circuits{
		Breakers:                 make(map[string]*CircuitBreaker),
		BreakersMutex:            &sync.RWMutex{},
		Settings:                 newSettingsCollection(),
		InitMetricCollectorFuncs: initMetricCollectorFuncs,
	}
}

func (c *Circuits) Configure(cmds map[string]CommandConfig) {
	for k, v := range cmds {
		c.ConfigureCommand(k, v)
	}
}

func (c *Circuits) ConfigureCommand(name string, config CommandConfig) {
	c.Settings.ConfigureCommand(name, config)
}

// Do runs your function in a synchronous manner, blocking until either your function succeeds
// or an error is returned, including hystrix circuit errors
func (c *Circuits) Do(name string, run runFunc, fallback fallbackFunc) error {
	done := make(chan struct{}, 1)

	r := func() error {
		err := run()
		if err != nil {
			return err
		}

		done <- struct{}{}
		return nil
	}

	f := func(e error) error {
		err := fallback(e)
		if err != nil {
			return err
		}

		done <- struct{}{}
		return nil
	}

	var errChan chan error
	if fallback == nil {
		errChan = c.Go(name, r, nil)
	} else {
		errChan = c.Go(name, r, f)
	}

	select {
	case <-done:
		return nil
	case err := <-errChan:
		return err
	}
}

// Go runs your function while tracking the health of previous calls to it.
// If your function begins slowing down or failing repeatedly, we will block
// new calls to it for you to give the dependent service time to repair.
//
// Define a fallback function if you want to define some code to execute during outages.
func (c *Circuits) Go(name string, run runFunc, fallback fallbackFunc) chan error {
	cmd := &command{
		run:          run,
		fallback:     fallback,
		start:        time.Now(),
		errChan:      make(chan error, 1),
		finished:     make(chan bool, 1),
		fallbackOnce: &sync.Once{},
	}

	// dont have methods with explicit params and returns
	// let data come in and out naturally, like with any closure
	// explicit error return to give place for us to kill switch the operation (fallback)

	circuit, _, err := c.getCircuit(name)
	if err != nil {
		cmd.errChan <- err
		return cmd.errChan
	}
	cmd.circuit = circuit

	go func() {
		defer func() { cmd.finished <- true }()

		// Circuits get opened when recent executions have shown to have a high error rate.
		// Rejecting new executions allows backends to recover, and the circuit will allow
		// new traffic when it feels a healthly state has returned.
		if !cmd.circuit.AllowRequest() {
			cmd.errorWithFallback(ErrCircuitOpen)
			return
		}

		// As backends falter, requests take longer but don't always fail.
		//
		// When requests slow down but the incoming rate of requests stays the same, you have to
		// run more at a time to keep up. By controlling concurrency during these situations, you can
		// shed load which accumulates due to the increasing ratio of active commands to incoming requests.
		cmd.Lock()
		select {
		case cmd.ticket = <-circuit.executorPool.Tickets:
			cmd.Unlock()
		default:
			cmd.Unlock()
			cmd.errorWithFallback(ErrMaxConcurrency)
			return
		}

		runStart := time.Now()
		runErr := run()

		if !cmd.isTimedOut() {
			cmd.runDuration = time.Since(runStart)

			if runErr != nil {
				cmd.errorWithFallback(runErr)
				return
			}

			cmd.reportEvent("success")
		}
	}()

	go func() {
		defer func() {
			cmd.Lock()
			cmd.circuit.executorPool.Return(cmd.ticket)
			cmd.Unlock()

			err := cmd.circuit.ReportEvent(cmd.events, cmd.start, cmd.runDuration)
			if err != nil {
				log.Print(err)
			}
		}()

		timer := time.NewTimer(c.getSettings(name).Timeout)
		defer timer.Stop()

		select {
		case <-cmd.finished:
		case <-timer.C:
			cmd.Lock()
			cmd.timedOut = true
			cmd.Unlock()

			cmd.errorWithFallback(ErrTimeout)
			return
		}
	}()

	return cmd.errChan
}

func (c *Circuits) initializeMetricCollectors(name string) []metricCollector.MetricCollector {

	metricCollectors := make([]metricCollector.MetricCollector, len(c.InitMetricCollectorFuncs), 0)

	for _, initFunc := range c.InitMetricCollectorFuncs {
		metricCollectors = append(metricCollectors, initFunc(name))
	}

	return metricCollectors
}

func (c *Circuits) getSettings(name string) *settings {
	return c.Settings.getSettings(name)
}

// Flush purges all circuit and metric information from memory.
// TODO: Evaluate whether this is necessary
func (c *Circuits) flush() {
	c.BreakersMutex.Lock()
	defer c.BreakersMutex.Unlock()

	for name, cb := range c.Breakers {
		cb.metrics.Reset()
		cb.executorPool.Metrics.Reset()
		delete(c.Breakers, name)
	}
}

// TODO: Evaluate whether this method is necessary
func (c *Circuits) getCircuitSettings() map[string]*settings {
	return c.Settings.GetCircuitSettings()
}

// GetCircuit returns the circuit for the given command and whether this call created it.
// TODO: Evaluate whether this is necessary
func (c *Circuits) getCircuit(name string) (*CircuitBreaker, bool, error) {
	c.BreakersMutex.RLock()
	_, ok := c.Breakers[name]
	if !ok {
		c.BreakersMutex.RUnlock()
		c.BreakersMutex.Lock()
		defer c.BreakersMutex.Unlock()
		// because we released the rlock before we obtained the exclusive lock,
		// we need to double check that some other thread didn't beat us to
		// creation.
		if cb, innerOK := c.Breakers[name]; innerOK {
			return cb, false, nil
		}
		c.Breakers[name] = newCircuitBreaker(name, c.Settings, c.initializeMetricCollectors(name))
	} else {
		defer c.BreakersMutex.RUnlock()
	}

	return c.Breakers[name], !ok, nil
}

func init() {
	//defaultCircuits = NewCircuits()
}

// Go runs your function while tracking the health of previous calls to it.
// If your function begins slowing down or failing repeatedly, we will block
// new calls to it for you to give the dependent service time to repair.
//
// Define a fallback function if you want to define some code to execute during outages.
func Go(name string, run runFunc, fallback fallbackFunc) chan error {
	return defaultCircuits.Go(name, run, fallback)
}

// Do runs your function in a synchronous manner, blocking until either your function succeeds
// or an error is returned, including hystrix circuit errors
func Do(name string, run runFunc, fallback fallbackFunc) error {
	return defaultCircuits.Do(name, run, fallback)
}

// GetCircuit returns the circuit for the given command and whether this call created it.
func GetCircuit(name string) (*CircuitBreaker, bool, error) {
	return defaultCircuits.getCircuit(name)
}

// Flush purges all circuit and metric information from memory.
func Flush() {
	defaultCircuits.flush()
}

// Configure applies settings for a set of circuits
func Configure(cmds map[string]CommandConfig) {
	defaultCircuits.Configure(cmds)
}

// ConfigureCommand applies settings for a circuit
func ConfigureCommand(name string, config CommandConfig) {
	defaultCircuits.ConfigureCommand(name, config)
}

func GetCircuitSettings() map[string]*settings {
	return defaultCircuits.getCircuitSettings()
}

func getSettings(name string) *settings {
	return defaultCircuits.getSettings(name)
}
