package hystrix

import (
	"sync"
	"time"
)

type settings struct {
	Timeout                time.Duration
	MaxConcurrentRequests  int
	RequestVolumeThreshold uint64
	SleepWindow            time.Duration
	ErrorPercentThreshold  int
}

type settingsCollection struct {
	Settings      map[string]*settings
	SettingsMutex *sync.RWMutex
}

func newSettingsCollection() *settingsCollection {
	return &settingsCollection{
		Settings:      make(map[string]*settings),
		SettingsMutex: &sync.RWMutex{},
	}
}

func (c *settingsCollection) ConfigureCommand(name string, config CommandConfig) {
	c.SettingsMutex.Lock()
	defer c.SettingsMutex.Unlock()

	timeout := DefaultTimeout
	if config.Timeout != 0 {
		timeout = config.Timeout
	}

	max := DefaultMaxConcurrent
	if config.MaxConcurrentRequests != 0 {
		max = config.MaxConcurrentRequests
	}

	volume := DefaultVolumeThreshold
	if config.RequestVolumeThreshold != 0 {
		volume = config.RequestVolumeThreshold
	}

	sleep := DefaultSleepWindow
	if config.SleepWindow != 0 {
		sleep = config.SleepWindow
	}

	errorPercent := DefaultErrorPercentThreshold
	if config.ErrorPercentThreshold != 0 {
		errorPercent = config.ErrorPercentThreshold
	}

	c.Settings[name] = &settings{
		Timeout:                time.Duration(timeout) * time.Millisecond,
		MaxConcurrentRequests:  max,
		RequestVolumeThreshold: uint64(volume),
		SleepWindow:            time.Duration(sleep) * time.Millisecond,
		ErrorPercentThreshold:  errorPercent,
	}
}

func (c *settingsCollection) GetCircuitSettings() map[string]*settings {
	copy := make(map[string]*settings)

	c.SettingsMutex.RLock()
	for key, val := range c.Settings {
		copy[key] = val
	}
	c.SettingsMutex.RUnlock()

	return copy
}

func (c *settingsCollection) getSettings(name string) *settings {
	c.SettingsMutex.RLock()
	s, exists := c.Settings[name]
	c.SettingsMutex.RUnlock()

	if !exists {
		c.ConfigureCommand(name, CommandConfig{})
		s = c.getSettings(name)
	}

	return s
}
