package twirpjackup3

import (
	"context"
	"math/rand"
	"strings"
	"time"

	"code.justin.tv/feeds/jackup3/proto/jackup3"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/sse/malachai/pkg/s2s/callee"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awsutil"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
)

var _ jackup3.Jackup3 = &Jackup3Impl{}

// Jackup3Impl implements our example app twirp interface
type Jackup3Impl struct {
	SecretKey        string
	ItemGetTableName string

	DynamoDB          *dynamodb.DynamoDB
	DownstreamCircuit *circuit.Circuit
	DynamoCircuit     *circuit.Circuit
}

// ItemGet fetches an item from (nothing) and stores current time in dynamodb
func (b *Jackup3Impl) ItemGet(ctx context.Context, req *jackup3.ItemGetRequest) (*jackup3.ItemGetResponse, error) {
	authInfo := awsutil.Prettify(callee.GetCallerID(ctx)) + awsutil.Prettify(callee.GetCapabilities(ctx))
	if req.Key == b.SecretKey {
		return &jackup3.ItemGetResponse{
			Value:    "YOU GUESSED THE SECRET KEY",
			AuthInfo: authInfo,
		}, nil
	}

	resp := make([]string, 0, 10)
	if err := b.DownstreamCircuit.Run(ctx, func(ctx context.Context) error {
		if strings.Contains(req.Key, "-downstream_failure-") {
			return errors.New("a downstream failure")
		}
		time.Sleep(time.Duration(rand.Float64() * float64(time.Millisecond*100)))
		return nil
	}); err != nil {
		return nil, err
	}

	putReq, _ := b.DynamoDB.PutItemRequest(&dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"id": {
				S: aws.String(req.GetKey()),
			},
			"setat": {
				S: aws.String(time.Now().String()),
			},
		},
		TableName: aws.String(b.ItemGetTableName),
	})
	if err := b.DynamoCircuit.Run(ctx, func(ctx context.Context) error {
		putReq.SetContext(ctx)
		return putReq.Send()
	}); err != nil {
		return nil, err
	}
	return &jackup3.ItemGetResponse{
		Value:    strings.Join(resp, ","),
		AuthInfo: authInfo,
	}, nil
}

// HealthCheck verifies the app works
func (b *Jackup3Impl) HealthCheck() error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*3)
	defer cancel()
	_, err := b.ItemGet(ctx, &jackup3.ItemGetRequest{
		Key: "test_key",
	})
	return err
}
