// +build integration

package main

import (
	"context"
	"errors"
	"fmt"
	"net"
	"net/http"
	"os"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/feeds/jackup3/proto/jackup3"
	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/servicerunner"
	"code.justin.tv/hygienic/twitchbaseservice/stacks5"
	"github.com/stretchr/testify/require"
)

func TestIntegrationGuessPassword(t *testing.T) {
	t.Parallel()
	s := setupService(t)
	if s == nil {
		t.Error("unable to setup service")
		return
	}

	client := jackup3.NewJackup3ProtobufClient(fmt.Sprintf("http://%s", s.OnListenService.TCPAddr()), http.DefaultClient)
	ctx := context.Background()
	t.Log("Sending req")
	resp, err := client.ItemGet(ctx, &jackup3.ItemGetRequest{
		Key: "abc",
	})
	require.Nil(t, err)
	require.Equal(t, "YOU GUESSED THE SECRET KEY", resp.Value)
	require.Nil(t, s.OnListenService.Close())
}

func TestIntegrationFreshFetch(t *testing.T) {
	t.Parallel()
	s := setupService(t)
	if s == nil {
		t.Error("unable to setup service")
		return
	}
	client := jackup3.NewJackup3ProtobufClient(fmt.Sprintf("http://%s", s.OnListenService.TCPAddr()), http.DefaultClient)
	ctx := context.Background()
	t.Log("Sending req")
	resp, err := client.ItemGet(ctx, &jackup3.ItemGetRequest{
		Key: time.Now().String(),
	})
	require.Nil(t, err)
	require.Equal(t, "", resp.Value)
	require.Nil(t, s.OnListenService.Close())
}

func TestIntegrationRepeatedFetch(t *testing.T) {
	t.Parallel()
	s := setupService(t)
	if s == nil {
		t.Error("unable to setup service")
		return
	}
	client := jackup3.NewJackup3ProtobufClient(fmt.Sprintf("http://%s", s.OnListenService.TCPAddr()), http.DefaultClient)
	ctx := context.Background()
	t.Log("Sending req")
	_, err := client.ItemGet(ctx, &jackup3.ItemGetRequest{
		Key: "123",
	})
	require.Nil(t, err)
	resp, err := client.ItemGet(ctx, &jackup3.ItemGetRequest{
		Key: "123",
	})
	require.Nil(t, err)
	require.Contains(t, resp.Value, "")
	require.Nil(t, s.OnListenService.Close())
}

type OnListenService struct {
	ListenAddr     net.Addr
	mainFinished   chan struct{}
	sigChan        chan os.Signal
	serviceAddress chan net.Addr
	exitCode       chan int
}

func NewOnListenTest(sigChan chan os.Signal) *OnListenService {
	return &OnListenService{
		mainFinished:   make(chan struct{}),
		sigChan:        sigChan,
		serviceAddress: make(chan net.Addr, 1),
		exitCode:       make(chan int, 1),
	}
}

func (s *OnListenService) Close() error {
	s.sigChan <- syscall.SIGTERM
	select {
	case <-s.mainFinished:
		return nil
	case <-time.After(time.Second * 100):
		return errors.New("service could not stop in time")
	}
}

func (s *OnListenService) OnListen(serviceAddr net.Addr) {
	s.serviceAddress <- serviceAddr
}

func (s *OnListenService) TCPAddr() string {
	l := s.ListenAddr.(*net.TCPAddr)
	return fmt.Sprintf("%s:%d", "localhost", l.Port)
}

func (s *OnListenService) OnExit(i int) {
	s.exitCode <- i
}

func (s *OnListenService) AssertExitCode(t *testing.T, expectedCode int) {
	select {
	case ec := <-s.exitCode:
		if ec != expectedCode {
			t.Errorf("Bad error code (expected vs actual) (%d vs %d)", expectedCode, ec)
		}
	case <-time.After(time.Second * 100):
		t.Error("took too long waiting for exit code")
	}
}

func (s *OnListenService) setupTime() time.Duration {
	return time.Second * 100
}

func (s *OnListenService) WaitForListenAddr() error {
	select {
	case <-s.mainFinished:
		return errors.New("main function finished before we could get listen address")
	case retAddr := <-s.serviceAddress:
		s.ListenAddr = retAddr
		return nil
	case <-time.After(s.setupTime()):
		return errors.New("took too long for service to startup")
	}
}

func (s *OnListenService) StartAndWaitForListenAddr(mainFunc func()) error {
	go func() {
		defer close(s.mainFinished)
		mainFunc()
	}()
	return s.WaitForListenAddr()
}

type setUpService struct {
	OnListenService *OnListenService
	service         service
}

func setupService(t *testing.T) *setUpService {
	sigChan := make(chan os.Signal)
	ret := NewOnListenTest(sigChan)
	var mem distconf.InMemory
	require.Nil(t, mem.Write("listen_addr", []byte(":0")))
	s := service{
		OnListen: ret.OnListen,
		osExit:   ret.OnExit,
		runner: servicerunner.ServiceRunner{
			SigChan: sigChan,
		},
		optionalConfig: stacks5.OptionalConfig{
			ForcedLogger: t,
			DefaultReaders: []distconf.Reader{
				&mem,
			},
			IgnoreDebugService: true,
		},
	}
	if err := ret.StartAndWaitForListenAddr(s.main); err != nil {
		t.Error(err)
		return nil
	}
	return &setUpService{
		OnListenService: ret,
		service:         s,
	}
}
