#!/bin/bash

function root__bootstrap() {
  if [ "${1-}" == "help" ]; then
    echo "Runs bootstrapping commands for when you first start"
    exit 0
  fi
  process_build bootstrap__ $@
}

function bootstrap__precheck() {
  # TODO: Check docker version >= 18.03
  if [ "${1-}" == "help" ]; then
    echo "Verify everything is ok to start"
    exit 0
  fi
  if [ "${SERVICE}" == "examp""leapp" ]; then
    echo "Please change ${SERVICE} to your service name"
    exit 1
  fi

  if [ -z ${SERVICE} ]; then
    echo "please input a service name"
    exit 1
  fi
  if [[ ! ${SERVICE} =~ ^[a-z][a-z][a-z0-9]+$ ]]; then
    echo "new service name must match regex [a-z][a-z][a-z]+"
    echo "you can modify this later, but a simple service name lets the bootstrap work without issues"
    echo "this regex is restrictive because of so many places the service name exists during bootstrapping"
    echo "it needs to be valid DynamoDB names, valid protobuf values, valid Go struct names, etc"
    exit 1
  fi
  which curl
  # TODO: Use https://stackoverflow.com/questions/4023830/how-to-compare-two-strings-in-dot-separated-version-format-in-bash
  docker version
  git version
  which tr
  aws --version
  echo "trying to assume pipeline role"
  (
    infra_assume_bootstrap_role pipeline
    aws sts get-caller-identity
    if [ $(aws_account_id) != ${OPS_PIPELINE_ACCOUNT_ID} ]; then
      echo "It appears your OPS_PIPELINE_ACCOUNT_ID variable is incorrect"
      echo "You set:  ${OPS_PIPELINE_ACCOUNT_ID}"
      echo "detected: $(aws_account_id)"
      exit 1
    fi

    pipeline_user=$(jq -r '.Parameters[0].ParameterValue' < cloudformation/service/service-pipeline.json | grep -o '[^/]*$')
    echo "Verifying existance of pipeline user ${pipeline_user}"
    echo "Detected pipeline user ${pipeline_user}"
    if ! aws iam get-user --user-name ${pipeline_user}; then
      echo "Unable to verify that user ${pipeline_user} exists"
      echo "Make sure you have ran https://git-aws.internal.justin.tv/twitch/jenkins_cf_user and uploaded the correct Jenkins secret file"
      exit 1
    fi
  )
  echo "trying to assume staging role"
  (
    infra_assume_bootstrap_role staging
    aws sts get-caller-identity
    if [ $(aws_account_id) != ${DEV_ACCOUNT_ID} ]; then
      echo "It appears your DEV_ACCOUNT_ID variable is incorrect"
      echo "You set:  ${DEV_ACCOUNT_ID}"
      echo "detected: $(aws_account_id)"
      exit 1
    fi
  )
  echo "trying to assume production role"
  (
    infra_assume_bootstrap_role production
    aws sts get-caller-identity
    if [ $(aws_account_id) != ${PROD_ACCOUNT_ID} ]; then
      echo "It appears your PROD_ACCOUNT_ID variable is incorrect"
      echo "You set:  ${PROD_ACCOUNT_ID}"
      echo "detected: $(aws_account_id)"
      exit 1
    fi
  )
  CUR=$(pwd)
  EXPECTED="${GOPATH}/src/${REPO}"
  if [ ${CUR} != ${EXPECTED} ]; then
    echo "You appear to be inside directory ${CUR}"
    echo "Your make.sh has REPO=${REPO}"
    echo "That implies you should be inside directory ${EXPECTED} since GOPATH is ${GOPATH}"
    echo "Please make sure your REPO variable is the location of your code repository relative to GOPATH/src and you are inside the correct directory"
    exit 1
  fi
  echo "Trying to connect to bastion"
  if ! USE_BUILDER=false make_run bastion execute whoami; then
    echo "Unable to connect to bastion host.  Verify that BASTION_ENV variable is setup correctly."
    echo "Or set your bastion host IP explicitly with BASTION_HOST_IP variable"
  fi
  echo "Everything is OK to start!"
}

function populate_network_vars() {
  if [ "${STAGING_VPC_ID}" == "GUESS" ]; then
    if ! STAGING_VPC_ID=$(infra_assume_bootstrap_role staging;guess_vpc_id); then
      echo "Unable to guess staging VPC ID: ${STAGING_VPC_ID}"
      exit 1
    fi
  fi
  if [ "${STAGING_SECURITY_GROUP}" == "GUESS" ]; then
    if ! STAGING_SECURITY_GROUP=$(infra_assume_bootstrap_role staging;guess_security_group_id); then
      echo "Unable to guess staging security groups ID: ${STAGING_SECURITY_GROUP}"
      exit 1
    fi
  fi
  if [ "${STAGING_SUBNETS}" == "GUESS" ]; then
    if ! STAGING_SUBNETS=$(infra_assume_bootstrap_role staging;guess_private_subnet_id ${STAGING_VPC_ID}); then
      echo "Unable to guess staging subnets: ${STAGING_SUBNETS}"
      exit 1
    fi
  fi

  if [ "${PROD_VPC_ID}" == "GUESS" ]; then
    if ! PROD_VPC_ID=$(infra_assume_bootstrap_role production;guess_vpc_id); then
      echo "Unable to guess prod vpc ID: ${PROD_VPC_ID}"
      exit 1
    fi
  fi
  if [ "${PROD_SECURITY_GROUP}" == "GUESS" ]; then
    if ! PROD_SECURITY_GROUP=$(infra_assume_bootstrap_role production;guess_security_group_id); then
      echo "Unable to guess prod security groups: ${PROD_SECURITY_GROUP}"
      exit 1
    fi
  fi
  if [ "${PROD_SUBNETS}" == "GUESS" ]; then
    if ! PROD_SUBNETS=$(infra_assume_bootstrap_role production;guess_private_subnet_id ${PROD_VPC_ID}); then
      echo "Unable to guess prod subnets: ${PROD_SUBNETS}"
      exit 1
    fi
  fi
}

function bootstrap__everything() {
  if [ "${1-}" == "help" ]; then
    echo "Ideally the *only* command you need to run to bootstrap everything!"
    exit 0
  fi
  TORUN=(
    "./make.sh bootstrap precheck"
    "./make.sh builder make"
    "./make.sh bootstrap rename_things"
    "git push --force origin HEAD:master"
    "./make.sh infra pipeline -a"
    "./make.sh infra update -a staging"
    "./make.sh infra update -a production"
    # TODO: Update README.md links with direct links to your LB
    "echo "" >> README.md"
    "git commit -am 'Small changes'"
    "git push --force origin HEAD:master"
  )

  idx=0
  for i in "${TORUN[@]}"; do
    idx=$((idx + 1))
    logfile="$TMPDIR/bootstrap_log${idx}.log"
    echo "Running ${i}.  Logfile ${logfile}"
    # Speak which step we're at
    say "${i}" || true
    if ! eval ${i} 2>&1 | tee ${logfile} ; then
      echo "Unable to run ${i}"
      echo "Check ${logfile} for details"
      exit 1
    fi
  done
  echo "DONE!"
  echo "Try the development workflow now!"
  echo "And check out your jenkins pipeline!"
  say "Bootstrapping complete!" || true
}

EXAMP_STAGING_VPC_ID="vpc-0effd93b771""8c080c"
EXAMP_STAGING_SECURITY_GROUP="sg-0fc6c""ea4177f1c0f2"
EXAMP_STAGING_SUBNETS="subnet-0e7d3c40f""d785d9fe,subnet-0bbcd91f57768ceb0,subnet-05be571e9ce5b7cc2"

EXAMP_PROD_VPC_ID="vpc-0b09a54a5""60dabb74"
EXAMP_PROD_SECURITY_GROUP="sg-0109""eb1d09b7aaaff"
EXAMP_PROD_SUBNETS="subnet-004e57fdc""4f2f6a78,subnet-00e1bdbdb97fd9c95,subnet-0f9861d12a0f2206a"

function git_replace() {
  echo "search replacing ${1} with ${2}"
  # Git search then globally replace $1 with $2
  git grep -l ${1} | xargs sed  -i "" -e "s#${1}#${2}#g" || true
}

# Note: REMOVE THIS WHEN DONE
function bootstrap__rename_things() {
  if [ "${1-}" == "help" ]; then
    echo "Rename to another service. You should start with a service that matches the regex [a-z]+"
    exit 0
  fi
  populate_network_vars
  # Break this appart so it does not get find/replaced
  base_service_name="exampl""eapp"
  base_service_name_caps="Exampl""eapp"
  base_dev_account_id="13161""2142357"
  base_jenkins_creds="twitch-graf""fiti-prod"
  base_prod_account_id="569048""605499"
  old_base_ghe="discov""ery/exam""pleapp"
  old_repo="code.justin.tv/dis""covery/ex""ampleapp"
  old_jenkins_job="job/disc""overy/job/exampl""eapp"
  new_service_base=$(echo ${REPO} | sed "s#code.justin.tv/##g")
  new_service_repo=$(echo "${new_service_base}" | sed 's#.*/##g')
  new_service_org=$(echo "${new_service_base}" | sed 's#/.*##g')

  if [ "${SERVICE}" == ${base_service_name} ]; then
    echo "Please change ${SERVICE} to your service name"
    exit 1
  fi

  newservice="${SERVICE}"
  newservice_upper="$(tr '[:lower:]' '[:upper:]' <<< ${newservice:0:1})${newservice:1}"
  if [ -z $newservice ]; then
    echo "please input a service name"
    exit 1
  fi
  if [[ ! ${newservice} =~ ^[a-z][a-z][a-z0-9]+$ ]]; then
    echo "new service name must match regex [a-z][a-z][a-z]+"
    echo "you can modify this later, but a simple service name lets the bootstrap work without issues"
    exit 1
  fi

  # This appears to be broken.  I don't think the parameters are escaping correctly
  git_replace "search-dis""covery" "${BOOTSTRAP_TEAM}"
  git_replace "${old_jenkins_job}" "job/${new_service_org}/job/${new_service_repo}"
  git_replace "${base_jenkins_creds}" "${OPS_PIPELINE_PROFILE}"
  git_replace "${old_base_ghe}" "${new_service_base}"
  git_replace "${old_repo}" "${REPO}"
  git_replace "${base_dev_account_id}" "${DEV_ACCOUNT_ID}"
  git_replace "${base_prod_account_id}" "${PROD_ACCOUNT_ID}"
  git_replace "${base_service_name}" "${newservice}"
  git_replace "${base_service_name_caps}" "${newservice_upper}"

  git_replace "${EXAMP_STAGING_VPC_ID}" "${STAGING_VPC_ID}"
  git_replace "${EXAMP_STAGING_SECURITY_GROUP}" "${STAGING_SECURITY_GROUP}"
  git_replace "${EXAMP_STAGING_SUBNETS}" "${STAGING_SUBNETS}"

  git_replace "${EXAMP_PROD_VPC_ID}" "${PROD_VPC_ID}"
  git_replace "${EXAMP_PROD_SECURITY_GROUP}" "${PROD_SECURITY_GROUP}"
  git_replace "${EXAMP_PROD_SUBNETS}" "${PROD_SUBNETS}"

  # This is for integration.run.json, which looks like "subnet-0bf0ec17e815d0396","subnet-04292ba0b31b7a811","subnet-083819452c6958bf7"
  OLD_SUBNET_ARR='"'$(echo ${EXAMP_STAGING_SUBNETS} | sed -e 's#,#","#g')'"'
  NEW_SUBNET_ARR='"'$(echo ${STAGING_SUBNETS} | sed -e 's#,#","#g')'"'

  git_replace "${OLD_SUBNET_ARR}" "${NEW_SUBNET_ARR}"

  echo "Renaming files"
  if [ -f cmd/${base_service_name}/internal/twirp${base_service_name}/${base_service_name}.go ]; then
    mv cmd/${base_service_name}/internal/twirp${base_service_name}/${base_service_name}.go cmd/${base_service_name}/internal/twirp${base_service_name}/${newservice}.go
  fi

  if [ -f cmd/${base_service_name}/internal/twirp${base_service_name}/${base_service_name}_test.go ]; then
    mv cmd/${base_service_name}/internal/twirp${base_service_name}/${base_service_name}_test.go cmd/${base_service_name}/internal/twirp${base_service_name}/${newservice}_test.go
  fi

  if [ -d cmd/${base_service_name}/internal/twirp${base_service_name} ]; then
    mv cmd/${base_service_name}/internal/twirp${base_service_name} cmd/${base_service_name}/internal/twirp${newservice}
  fi

  if [ -d cmd/${base_service_name} ]; then
    mv cmd/${base_service_name} cmd/${newservice}
  fi

  rm -rf ./proto/${base_service_name}
  mkdir -p ./proto/${newservice}

  if [ -f proto/${base_service_name}.proto ]; then
    mv proto/${base_service_name}.proto proto/${newservice}.proto
  fi


  echo "Creating builder"
  make_run builder make
  echo "Running code generation"
  make_run extra generate
  echo "Reformatting your code since variables changed"
  make_run go format
  echo "Building your go code"
  make_run go build ./...
  echo "Testing your go code"
  make_run go test ./...

  echo "Committing your code to git"
  git add .
  git commit -am "Renamed everything" || true
  git remote remove ${base_service_name} || true
  echo "Everything is ready to go!  Push your code."
  echo "   git push --force origin HEAD:master"
  echo "You can also remove bootstrapping"
}
