#!/bin/bash

function root__builder() {
  if [ "${1-}" == "help" ]; then
    echo "Runs builder image related commands"
    exit 0
  fi
  process_build builder__ $@
}

function builder__push() {
  if [ "${1-}" == "help" ]; then
    echo "Push a created builder image"
    exit 0
  fi
  (
    if in_jenkins; then
      infra_assume_pipeline_role
    else
      infra_assume_bootstrap_role pipeline
    fi
    ecr_push_core ${BUILDER} ${BUILDER_TAG}
  )
}

function git_directory_of_clone() {
  echo "${1}"| sed 's#.*/##g' | sed 's#.git$##g'
}

function builder_checkout_version() {
  REPO=${1-}
  if [[ -z ${REPO} ]]; then
    echo "Please set a repository to checkout"
  fi
  VERSION=${2-}
  (
    cd builder
    DIR=$(git_directory_of_clone ${REPO})
    [[ ! -z ${DIR} ]]
    if [[ -d ${DIR}/.git ]]; then
      # I really want to rm -rf here honestly, but I cannot since that's too dangerous against a bash variable
      # If you get to an error here, just delete the ${REPO} directory that was left over
      mv ${DIR} ${TMPDIR-/tmp}/${RANDOM}${DIR}
    fi
    git clone ${REPO}
    # This allows us to reset to tags that may not be on the master branch
    (cd $(git_directory_of_clone ${REPO}) && git fetch -av)
    if [[ ! -z ${VERSION} ]]; then
      # We have to do it this way for people that use submodules since we want to reset the submodules to what they
      # were at *that* version
      cd $(git_directory_of_clone ${REPO})
      git reset --hard v${VERSION}
      git submodule init
      git submodule update --recursive --remote
    fi
  )
}

function builder__make() {
  if [ "${1-}" == "help" ]; then
    echo "Create the builder docker image"
    exit 0
  fi
  # I would love to put these `clone` steps inside the docker file, but I can't seem to get them working locally since
  # our local SSH uses teleport-bastion and that doesn't forward into `docker build`
  ( cd builder && rm -rf twirpserviceslohook sandstorm terraform-provider-grafana )
  builder_checkout_version git@git-aws.internal.justin.tv:hygienic/twirpserviceslohook.git
  builder_checkout_version git@git-aws.internal.justin.tv:twitch/terraform-provider-grafana.git
  builder_checkout_version git@git-aws.internal.justin.tv:systems/sandstorm.git 1.5.2
  (
    cd builder
    docker build --build-arg SANDSTORM_CLI=1.5.2 -t ${BUILDER} .
    rm -rf twirpserviceslohook sandstorm terraform-provider-grafana
  )
}

function builder__run() {
  if [ "${1-}" == "help" ]; then
    echo "Run this script inside the builder image, passing extra parameters to the build command"
    exit 0
  fi
  make_run builder download
  # https://docs.docker.com/docker-for-mac/networking/
  docker_args="-e socks_addr=host.docker.internal:4456 -e USE_BUILDER=false -v $(local_docker_cache_directory):/root/.cache"
  if [ ! -z "${GOPATH}" ]; then
    # Change the docker container's gopath to our gopath, so that when error messages show a filename that filename matches
    # what's on our local development machine
    docker_args="${docker_args} -e GOPATH=${GOPATH} -v ${GOPATH}/src:${GOPATH}/src:cached -w ${GOPATH}/src/${REPO}"
  fi

  if [ ! -z "${AWS_SHARED_CREDENTIALS_FILE-}" ]; then
#     Put the shared creds file inside the docker container so it can pretend to do AWS commands
    docker_args="${docker_args} -e AWS_SHARED_CREDENTIALS_FILE=/awsconfig -v ${AWS_SHARED_CREDENTIALS_FILE}:/awsconfig:ro"
  fi

  if [[ ! -z "${USE_TWINIT-}" ]] && kstatus; then
    docker_args="${docker_args} -e HTTPS_PROXY=http://host.docker.internal:${TWINIT_TUNNEL_LOCAL_PORT-12345} -e HTTP_PROXY=http://host.docker.internal:${TWINIT_TUNNEL_LOCAL_PORT-12345}"
  fi

  if in_jenkins; then
    # When running in jenkins, the current checkout should become our repository
    docker_args="${docker_args} -v $(pwd):/go/src/$REPO"
  else
    # Interactive only works when you have a TTY (IE not jenkins)
    docker_args="${docker_args} ${LOCAL_DOCKER_RUN_ARGS-}"
    if [ -z "${SKIP_TTY-}" ]; then
      # you may want to disable the TTY because of https://github.com/moby/moby/issues/8513
      docker_args="${docker_args} -it"
    fi
  fi

  # Some environment variables are needed by sub commands, if they are present
  PASS_THRU=(GIT_COMMIT DEBUG BUILD_ID AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY AWS_SESSION_TOKEN JENKINS_HOME TF_LOG HTTP_PROXY HTTPS_PROXY NO_PROXY RUNNER)
  for t in ${PASS_THRU[@]}; do
    if [ ! -z ${!t-} ]; then
      docker_args="${docker_args} -e $t=${!t}"
    fi
  done

  if [ -e /var/run/docker.sock ]; then
    # If we have docker locally, cross mount the docker socket so we can run docker commands
    docker_args="${docker_args} -v /var/run/docker.sock:/var/run/docker.sock"
  fi

  if [ -f $HOME/.midway/cookie ]; then
    docker_args="${docker_args} -v $HOME/.midway:/root/.midway:ro"
  fi
  # Do this to forward sigkill correctly to docker
  exec docker run ${docker_args} ${BUILDER} $@
}

function builder__push_if_needed() {
  if [ "${1-}" == "help" ]; then
    echo "Will build and push the builder if needed to"
    exit 0
  fi
  if make_run builder download; then
    return
  fi
  make_run builder make
  make_run builder push
}

function builder__download() {
  if [ "${1-}" == "help" ]; then
    echo "Download the builder image from the remote repository"
    exit 0
  fi
  if docker_image_exists_locally ${BUILDER} ; then
    return
  fi
  (
    infra_assume_infra_role staging
    ecr_pull ${BUILDER}
  )
}

function builder__shell() {
  if [ "${1-}" == "help" ]; then
    echo "Run a bash shell.  You can builder shell to get inside an interactive shell with the builder"
    exit 0
  fi
  if [ "${USE_BUILDER}" == "true" ]; then
    make_exec builder run builder shell $@
  fi
  bash
}
