#!/bin/bash

ISENGARD_CACHE_DIR="${ISENGARD_CACHE_DIR-"$HOME/.cache/isengard"}"

if [ "${DEBUG-}" = "true" ]; then
  set -x
fi

# Execute this make command in a new shell then return.  For example, to simulate `./make go test` you could call
# `make_run go test`
function make_run() {
  DIR="${BASH_SOURCE%/*}"
  if [[ ! -d "$DIR" ]]; then DIR="$PWD"; fi
  (
    cd ${DIR}/../..
    ./make.sh $@
  )
}

# Execute this make command replacing the current execution.  This is done via `exec`.  It does not return.  To simulate
# running `./make go test`, you can call `make_exec go test`
function make_exec() {
  DIR="${BASH_SOURCE%/*}"
  if [[ ! -d "$DIR" ]]; then DIR="$PWD"; fi
  cd ${DIR}/../..
  exec ./make.sh $@
}

# By default does nothing.  You can overwrite this function inside vars.sh to populate any environment variables your
# application needs when it runs locally
function populate_app_env() {
  return 0
}

function is_git_dirty {
  [[ $(git diff --shortstat 2> /dev/null | tail -n1) != "" ]]
}

# Expose env variables that allow us to run commands as an assumed role
function assume_role() {
  CREDS=$(aws sts assume-role $@ --query '[Credentials.AccessKeyId,Credentials.SecretAccessKey,Credentials.SessionToken]' --output text)
  if [ -z "${CREDS}" ]; then
    return 1
  fi
  read -ra CREDS_ARR <<< "$CREDS"
  export AWS_ACCESS_KEY_ID=${CREDS_ARR[0]}
  export AWS_SECRET_ACCESS_KEY=${CREDS_ARR[1]}
  export AWS_SESSION_TOKEN=${CREDS_ARR[2]}
}

function aws_account_id() {
  aws sts get-caller-identity --query 'Account' --output text
}

# Push Docker container to ECR
function ecr_push() {
  (
    INFRA=${1-}
    if [ -z ${INFRA} ]; then
      echo "First parameter required"
      exit 1
    fi
    DOCKER_TAG_PREFIX=${2-}
    ecr_push_core ${SERVICE}:${DOCKER_TAG_PREFIX}${GIT_COMMIT} ${SERVICE}-${INFRA}:${DOCKER_TAG_PREFIX}${GIT_COMMIT}
  )
}

# Returns true if the docker image exists locally
function docker_image_exists_locally() {
  IMAGE=${1-}
  if [ -z ${IMAGE} ]; then
    echo "First parameter required: Image to check for"
    exit 1
  fi
  docker image inspect ${IMAGE} > /dev/null
}

# Pull an image from ECR
function ecr_pull() {
  (
    REMOTE_TAG=${1-}
    DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${OPS_PIPELINE_ACCOUNT_ID})
    eval ${DOCKER_CMD}

    DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${DEV_ACCOUNT_ID})
    eval ${DOCKER_CMD}

    DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${PROD_ACCOUNT_ID})
    eval ${DOCKER_CMD}
    docker pull ${REMOTE_TAG}
  )
}

function ecr_push_core() {
  (
    ACCOUNT_ID=$(aws_account_id)
    LOCAL_TAG=${1-}
    REMOTE_TAG=${2-}
    REMOTE_DOCKER_NAME=${ACCOUNT_ID}.dkr.ecr.${AWS_DEFAULT_REGION}.amazonaws.com/${REMOTE_TAG}
    DOCKER_CMD=$(aws ecr get-login --no-include-email)
    eval ${DOCKER_CMD}
    docker tag ${LOCAL_TAG} ${REMOTE_DOCKER_NAME}
    docker push ${REMOTE_DOCKER_NAME}
  )
}

# Run protobuf, twirp, and protobuf linter
function default_protobuf_generation() {
  GOPATH=${GOPATH-}
  if [ -z $GOPATH ]; then
    echo "GOPATH required for build"
    exit 1
  fi
  protoc --lint_out=. --proto_path=${GOPATH}/src:. --twirp_out=${GOPATH}/src --go_out=${GOPATH}/src proto/${SERVICE}.proto
}

# Combination of prep_term/handle_term/wait_term allow forwarding SIG from bash into another command
# See https://unix.stackexchange.com/questions/146756/forward-sigterm-to-child-in-bash for why we need this
# Forwarding signals allows us to forward a SIGKILl to running docker containers from our bash script
function prep_term() {
    unset term_child_pid
    unset term_kill_needed
    trap 'handle_term' TERM INT
}
function handle_term() {
    if [ "${term_child_pid}" ]; then
        kill -TERM "${term_child_pid}" 2>/dev/null
        # The child process will probably send some RPC to clean itself up.  Give it a tiny amount of time to actually
        # do that.
        sleep 2
        # Wait for the child process to finish killing itself
        wait ${term_child_pid}
    else
        term_kill_needed="yes"
    fi
}
function wait_term() {
    term_child_pid=$!
    if [ "${term_kill_needed-}" ]; then
        kill -TERM "${term_child_pid}" 2>/dev/null
    fi
    wait ${term_child_pid}
    trap - TERM INT
    wait ${term_child_pid}
}

# Process a help flag for every subcommand
function process_help() {
  prefix=$1
  shift
  if [ "${1-}" == "help" ]; then
    echo "Prints each command and help"
    exit 0
  fi

  compgen -A function | grep $prefix | sort | while read -r line; do
    echo
    echo "  ${line:${#prefix}}"
    echo "    $($line help)"
  done
}

# Fetch credentials from isengard and populate ENV.  These creds are used by AWS SDK to authenticate.  Note that
# it uses a local file system cache to prevent spamming isengard.
# The location is ${ISENGARD_CACHE_DIR} (set above)
# $1 is the account ID and $2 is the role name
function isengard_get_credentials() {
  if [ ! -f ~/.midway/cookie ]; then
    echo "midway cookie not found.  You may need to run mwinit"
    exit 1
  fi

  ACCOUNT_ID=${1-}
  ROLE=${2-}
  mkdir -p ${ISENGARD_CACHE_DIR}
  ISENGARD_CACHE="${ISENGARD_CACHE_DIR}/${ACCOUNT_ID}-${ROLE}.json"
  # refresh cache at 10 minutes
  REFRESH_AT=${ISENGARD_CACHE_TIME-600000}
  if [ -f ${ISENGARD_CACHE} ]; then
    credentials=$(cat ${ISENGARD_CACHE})
    EXPIRATION="$(echo "$credentials" | jq -r '.expiration')";
    CURTIME=$(($(date +%s)*1000))
    LEFTOVER=$(($EXPIRATION-$CURTIME))
    if (( LEFTOVER > REFRESH_AT )); then
      echo ${credentials}
      return 0
    fi
    # Cache bad.  Kill it with fire
    rm -f ${ISENGARD_CACHE}
  fi
  credentials_and_code=$(curl --silent --write-out "HTTPSTATUS:%{http_code}" -k -b ~/.midway/cookie -c ~/.midway/cookie -L -X POST --header "X-Amz-Target: IsengardService.GetAssumeRoleCredentials" --header "Content-Encoding: amz-1.0" --header "Content-Type: application/json; charset=UTF-8" -d "{\"AWSAccountID\": \"$1\", \"IAMRoleName\":\"$2\"}" https://isengard-service.amazon.com);
  HTTP_STATUS=$(echo "${credentials_and_code}" | tr -d '\n' | sed -e 's/.*HTTPSTATUS://')
  HTTP_BODY=$(echo "${credentials_and_code}" | sed -e 's/HTTPSTATUS\:.*//g')
  if (( "200" != "${HTTP_STATUS}")); then
    echo "Invalid isengard status code: ${HTTP_STATUS}"
    echo "${HTTP_BODY}"
    exit 1
  fi
  credentials=$(echo "${HTTP_BODY}" | jq -r '.AssumeRoleResult |fromjson | .credentials')
  echo ${credentials} > ${ISENGARD_CACHE}
  echo ${credentials}
}

# Assume a role from isengard.  $1 is the account ID and $2 is the role name
function isengard_assume_role() {
  if [ ! -f ~/.midway/cookie ]; then
    echo "midway cookie not found.  You may need to run mwinit"
    exit 1
  fi

  if ! credentials=$(isengard_get_credentials $@); then
    echo "${credentials}"
    exit 1
  fi
  export AWS_ACCESS_KEY_ID="$(echo "$credentials" | jq -r '.accessKeyId')";
  export AWS_SESSION_TOKEN="$(echo "$credentials" | jq -r '.sessionToken')";
  export AWS_SECRET_ACCESS_KEY="$(echo "$credentials" | jq -r '.secretAccessKey')";
}

# Return true if this script is running inside jenkins
function in_jenkins() {
 [ ! -z "${JENKINS_HOME-}" ]
}

# Each builder has its own local copy of the ~/.cache directory that we do not throw away.  This allows us to
# do faster go build commands
function local_docker_cache_directory() {
  echo /tmp/$(sanitize_builder)
}

function sanitize_builder() {
  echo ${BUILDER} | sed -e 's/[^A-Za-z0-9._-]/_/g'
}

# Print out a cloudformation output value
function get_cloudformation_output() {
  VAR=${1-}
  if [ -z ${VAR} ]; then
    exit 1
  fi
  RES=$(aws cloudformation list-exports --no-paginate --query "Exports[?Name=='$1'].Value" --output text)
  if [ -z ${RES} ]; then
    exit 1
  fi
  echo ${RES}
}

# Print a cloudformation output variable, but caches it to disk so we don't have to go out each time
function cached_cloudformation_output() {
  VAR=${1-}
  if [ -z ${VAR} ]; then
    exit 1
  fi
  TD=${TMPDIR-/tmp}
  CACHE_FILE=${TD}/$(sanitize_builder)${VAR}
  if [ -f ${CACHE_FILE} ]; then
    cat ${CACHE_FILE}
    exit 0
  fi
  VAL=$(get_cloudformation_output ${VAR})
  if [ -z ${VAL} ]; then
    exit 1
  fi
  echo ${VAL} > ${CACHE_FILE}
  echo ${VAL}
}

# This is the core to our __ expansion magic
function process_build() {
  prefix=$1
  shift
  T="$prefix${1-}"
  if declare -F "$T" >/dev/null ; then
    func="$prefix${1}"
    shift; # pop $1
    "$func" "$@"    # invoke our named function w/ all remaining arguments
  else
    echo "Undefined subcommand ${1-}"
    process_help $prefix
    exit 1
  fi
}

# Assume the right bootstrap role for an environment
function infra_assume_bootstrap_role() {
  INFRA="${1-}"
  SESSION_NAME="bootstrap-${INFRA}-$(whoami)"
  if [ "${1-}" == "pipeline" ]; then
    isengard_assume_role ${OPS_PIPELINE_ACCOUNT_ID} ${OPS_PIPELINE_ROLE}
    return
  fi
  if [ "${1-}" == "staging" ]; then
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
    return
  fi
  if [[ "${1-}" == "personal" ]]; then
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
    return
  fi
  if [ "${1-}" == "production" ]; then
    isengard_assume_role ${PROD_ACCOUNT_ID} ${PROD_ISENGARD_ROLE}
    return
  fi
  echo "Unable to assume role for infra ${INFRA}.  Is this a valid infra name?"
  exit 1
}

# Assume the role applications run as inside an infrastructure
function assume_dev_role_from_infra() {
  SESSION_NAME="localrun-$(whoami)"
  assume_role --role-arn ${LOCAL_DEV_ROLE} --role-session-name ${SESSION_NAME}
}

function infra_assume_pipeline_role() {
  # Note: only supported in jenkins so far
  if in_jenkins; then
    AWS_PROFILE=${OPS_PIPELINE_PROFILE} assume_role --role-arn ${PIPELINE_RUNNING_ROLE} --role-session-name "role-from-jenkins-${BUILD_ID-}"
  fi
}

function personal_deploy_role() {
  echo "arn:aws:iam::${DEV_ACCOUNT_ID}:role/${SERVICE}-personal${RUNNER}-ops-deployer"
}

# Assume the deployment role inside an infrastructure
function infra_assume_infra_role() {
  INFRA="${1-}"
  SESSION_NAME="push-${INFRA}-$(whoami)"
  # If jenkins, assume pipeline first.  Otherwise try to assume the role directly
  infra_assume_pipeline_role

  # TODO: Abstract this better?

  if [ "${1-}" == "personal" ]; then
    isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
    assume_role --role-arn $(personal_deploy_role) --role-session-name ${SESSION_NAME}
    return
  fi

  if [ "${1-}" == "staging" ]; then
    if in_jenkins; then
      assume_role --role-arn ${DEV_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    else
      isengard_assume_role ${DEV_ACCOUNT_ID} ${DEV_ISENGARD_ROLE}
      assume_role --role-arn ${DEV_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    fi
    return
  fi
  if [ "${1-}" == "production" ]; then
    if in_jenkins; then
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    else
      isengard_assume_role ${PROD_ACCOUNT_ID} ${PROD_ISENGARD_ROLE}
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    fi
    return
  fi
  if [ "${1-}" == "canary" ]; then
    if in_jenkins; then
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    else
      isengard_assume_role ${PROD_ACCOUNT_ID} ${PROD_ISENGARD_ROLE}
      assume_role --role-arn ${PROD_DEPLOY_ROLE} --role-session-name ${SESSION_NAME}
    fi
    return
  fi
  echo "Unable to assume role for deployment ${INFRA}.  Is this a valid infra name?"
  exit 1
}
