#!/bin/bash

function root__extra() {
  if [ "${1-}" == "help" ]; then
    echo "Includes one off commands that don't really have a place"
    exit 0
  fi
  process_build extra__ $@
}

function extra__ok_for_development() {
  if [ "${1-}" == "help" ]; then
    echo "Verify you as a developer have everything setup to go"
    exit 0
  fi
  echo "=== Checking for installed versions ..."
  which curl
  which tr
  docker version
  git version
  aws --version
  echo "=== OK"
  echo

  echo "=== Checking ability to assume developer role"
  (
    infra_assume_bootstrap_role staging
    aws sts get-caller-identity
  )
  echo "=== OK"
  echo

  echo "=== Checking ability connect to bastion tunnel"
  make_run bastion execute whoami
  echo "=== OK"
  echo

  echo "=== Running unit tests"
  make_run go test
  echo "=== OK"
  echo

  echo "=== running integration tests"
  make_run go integration_test
  echo "=== OK"
  echo

  echo "=== running docker build"
  make_run docker build
  make_run docker build_integration
  echo "=== OK"
  echo
  echo "You are ok to develop code!"
}

function extra__cloudformation() {
  if [ "${1-}" == "help" ]; then
    echo "Runs cloudformation command without assuming any roles"
    exit 0
  fi
  if [ "${USE_BUILDER}" == "true" ]; then
    make_exec builder run extra cloudformation $@
  fi
  CFEXECUTE_OPTIONS="-clean"
  AUTO=""
  while getopts ":a" opt; do
    case ${opt} in
      a)
        CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -auto"
        AUTO="true"
        ;;
      \?)
        echo "Invalid option: -$OPTARG" >&2
        return 1
        ;;
    esac
  done
  shift "$((OPTIND-1))"

  STACK_LOC="${1-}"
  if [ -z ${STACK_LOC} ]; then
    echo "please specify a stack to update"
    exit 1
  fi
  if [ ! -f "${STACK_LOC}" ]; then
    echo "unable to find changeset file ${STACK_LOC}"
    exit 1
  fi
  CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -changeset ${STACK_LOC}"

  if [ "${DEBUG-}" = "true" ]; then
    CFEXECUTE_OPTIONS="${CFEXECUTE_OPTIONS} -verbosity 2"
  fi
  if [ ! -z ${AUTO} ]; then
    # prep_term and wait_term are how you forward signals into bash child processes
    prep_term
    cfexecute ${CFEXECUTE_OPTIONS} &
    wait_term
  else
    cfexecute ${CFEXECUTE_OPTIONS}
  fi
}

function extra__update_common_script() {
  if [ "${1-}" == "help" ]; then
    echo "Update common script files"
    exit 0
  fi
  git subtree pull --prefix scripts/jenkins_cf_deployer git@git-aws.internal.justin.tv:twitch/jenkins_cf_deployer.git master --squash
}

function extra__push_common_script() {
  if [ "${1-}" == "help" ]; then
    echo "Push a PR for common script files"
    exit 0
  fi
  git subtree push --prefix scripts/jenkins_cf_deployer git@git-aws.internal.justin.tv:twitch/jenkins_cf_deployer.git review-$(whoami) --squash
}

function extra__versions() {
  if [ "${1-}" == "help" ]; then
    echo "Prints version information for all used binaries"
    exit 0
  fi
  if [ "${USE_BUILDER}" == "true" ]; then
    make_exec builder run extra versions $@
  fi
  echo "Go version              $(go version)"
  echo "protoc version          $(protoc --version)"
  echo "gometalinter version    $(gometalinter --version)"
  echo "Twirp version           $(protoc-gen-twirp --version)"
  echo "Docker version          $(docker --version)"
  echo "aws cli version"
  aws --version
  echo "Dep version"
  dep version
}

function extra__generate() {
  if [ "${1-}" == "help" ]; then
    echo "Regenerate autogenerated files"
    exit 0
  fi
  if [ "${USE_BUILDER}" == "true" ]; then
    make_exec builder run extra generate $@
  fi
  default_protobuf_generation
  default_slo_generation
}

function default_slo_generation() {
  slogenerator proto/${SERVICE}/${SERVICE}.twirp.go ${SERVICE} > proto/${SERVICE}/slo.go
}

function extra__upload_code_coverage() {
  if [ "${1-}" == "help" ]; then
    echo "Upload previously generated code coverage into codecov service"
    exit 0
  fi
  # Take out code.justin.tv from the code coverage repo
  CODECOV_REPO=${REPO:15}
  curl -s https://codecov.internal.justin.tv/bash | bash -s - -r ${CODECOV_REPO}
}