package events

import (
	"encoding/json"
	"time"

	"code.justin.tv/sse/malachai/pkg/internal/closer"
	"code.justin.tv/sse/malachai/pkg/log"
)

// Buffer is a buffer of logs held locally before being sent into kinesis
type Buffer interface {
	Write([]byte) (int, error)

	// Run should start processing logs on a
	Run(reportInterval time.Duration, batchWriter FirehoseBatchWriter)
}

// Logger describes an event logger
type Logger interface {
	Run()
	WriteEvent(interface{})
	WithFields(DefaultableEvent) EventWriter
}

// Client writes events directly to firehose
type Client struct {
	Buffer          Buffer
	FirehoseBatcher FirehoseBatchWriter
	Logger          log.S2SLogger
	ReportInterval  time.Duration
	Closer          *closer.Closer
}

// Run starts the background updater
func (c *Client) Run() {
	if c.Closer == nil {
		c.Closer = closer.New()
	}
	c.Buffer.Run(c.ReportInterval, c.FirehoseBatcher)
}

// Close closes the client
func (c *Client) Close() error {
	return c.Closer.Close()
}

// WriteEvent writes a list of events to file
func (c *Client) WriteEvent(event interface{}) {
	var (
		raw []byte
		err error
	)

	if raw, err = json.Marshal(event); err != nil {
		c.Logger.Errorf("error marshaling event for logs: %s", err.Error())
	}

	if _, err = c.Buffer.Write(append(raw, []byte("\n")...)); err != nil {
		c.Logger.Errorf("error writing event to logs: %s", err.Error())
	}
}

// WithFields wraps this client in a client that provides defaults
func (c *Client) WithFields(defaults DefaultableEvent) EventWriter {
	return &WithFieldsEventWriter{
		EventWriter:   c,
		EventDefaults: defaults,
	}
}
