package twirpjlindamoup

import (
	"context"
	"strings"
	"time"

	"code.justin.tv/sse/malachai/pkg/s2s/callee"
	"github.com/aws/aws-sdk-go/aws/awsutil"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
	"github.com/go-redis/redis"

	"code.justin.tv/feeds/jlindamoup/proto/jlindamoup"
)

var _ jlindamoup.Jlindamoup = &JlindamoupImpl{}

// JlindamoupImpl implements our example app twirp interface
type JlindamoupImpl struct {
	SecretKey        string
	ItemGetTableName string

	DynamoDB      *dynamodb.DynamoDB
	Redis         *redis.Client
	RedisCircuit  *circuit.Circuit
	DynamoCircuit *circuit.Circuit
}

// ItemGet fetches an item from redis and stores current time in dynamodb
func (b *JlindamoupImpl) ItemGet(ctx context.Context, req *jlindamoup.ItemGetRequest) (*jlindamoup.ItemGetResponse, error) {
	authInfo := awsutil.Prettify(callee.GetCallerID(ctx)) + awsutil.Prettify(callee.GetCapabilities(ctx))
	if req.Key == b.SecretKey {
		return &jlindamoup.ItemGetResponse{
			Value:    "YOU GUESSED THE SECRET KEY",
			AuthInfo: authInfo,
		}, nil
	}

	resp := make([]string, 0, 10)
	if err := b.RedisCircuit.Run(ctx, func(ctx context.Context) error {
		if getRes := b.Redis.Get(req.Key); getRes.Err() != nil {
			if getRes.Err() == redis.Nil {
				resp = append(resp, "not found in redis")
			} else {
				return getRes.Err()
			}
		} else {
			resp = append(resp, "found in redis", getRes.String())
		}
		if setRes := b.Redis.Set(req.Key, time.Now().String(), time.Hour); setRes.Err() != nil {
			return setRes.Err()
		}
		return nil
	}); err != nil {
		return nil, err
	}

	putReq, _ := b.DynamoDB.PutItemRequest(&dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"id": {
				S: aws.String(req.GetKey()),
			},
			"setat": {
				S: aws.String(time.Now().String()),
			},
		},
		TableName: aws.String(b.ItemGetTableName),
	})
	if err := b.DynamoCircuit.Run(ctx, func(ctx context.Context) error {
		putReq.SetContext(ctx)
		return putReq.Send()
	}); err != nil {
		return nil, err
	}
	return &jlindamoup.ItemGetResponse{
		Value:    strings.Join(resp, ","),
		AuthInfo: authInfo,
	}, nil
}

// HealthCheck verifies the app works
func (b *JlindamoupImpl) HealthCheck() error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*3)
	defer cancel()
	_, err := b.ItemGet(ctx, &jlindamoup.ItemGetRequest{
		Key: "test_key",
	})
	return err
}
