#!/bin/bash
set -e

if [ ! -z "$DEBUG" ]; then
  set -x
fi

usage() {
  echo "usage: $0"
  echo "  Prints out to stdout guessed VPC information based upon how accounts are generally setup currently"
  echo "Required environment variables"
  echo "  AWS_PROFILE -- .aws/config profile that you're using for the push"
  echo "Optional environment variables"
  echo "  AWS_REGION    -- Which region to guess for.  Defaults to us-west-2"
}

AWS_REGION=${AWS_REGION:-us-west-2}

# Find the current account ID in AWS_PROFILE
aws_account_id() {
  aws --region ${AWS_REGION} --profile ${AWS_PROFILE} sts get-caller-identity --query 'Account' --output text
}

function guess_vpc_id() {
	aws --region ${AWS_REGION} --profile ${AWS_PROFILE} ec2 describe-vpcs --filters "Name=tag:Name,Values=${AWS_PROFILE}" --query 'Vpcs[].VpcId' --output text
}

function guess_security_group_id() {
  aws --region ${AWS_REGION} --profile ${AWS_PROFILE} ec2 describe-security-groups --query "SecurityGroups[?GroupName=='twitch_subnets'].GroupId" --output text
}

function guess_private_subnet_id() {
  VPC_ID=$1
  aws --region ${AWS_REGION} --profile ${AWS_PROFILE} ec2 describe-subnets --query "Subnets[? ! MapPublicIpOnLaunch] | [?VpcId =='$VPC_ID'].SubnetId | join(',', @)" --output text
}

if [ -z "${AWS_PROFILE}" ]; then
  echo "Please set ENV variable AWS_PROFILE.  It's usually the name of your AWS account you're deploying into"
  usage
  exit 1
fi

if [ ! -z "$@" ]; then
  usage
  exit 1
fi

VPC_ID=$(guess_vpc_id)
echo "AWS profile"
echo "  ${AWS_PROFILE}"
echo "AWS region"
echo "  ${AWS_REGION}"
echo "Account ID"
echo "  $(aws_account_id)"
echo "VPC id"
echo "  ${VPC_ID}"
echo "Security group"
echo "  $(guess_security_group_id)"
echo "Private Subnets"
echo "  $(guess_private_subnet_id ${VPC_ID})"
