terraform {
  backend "s3" {
    bucket = "jlindamoup-twitch-base-tf"
    key    = "terraform/jlindamoup"
    region = "us-west-2"
  }
}

// https://github.com/terraform-providers/terraform-provider-pagerduty
provider pagerduty {
  # Use explicit versions to prevent issues when the plugin changes
  version = "1.2"

  # Token set via ENV
}

// https://github.com/terraform-providers/terraform-provider-grafana
provider grafana {
  url     = "https://grafana.internal.justin.tv/"
  version = "v1.2.0-fiximportsbools"              # a custom version until https://github.com/terraform-providers/terraform-provider-grafana/pull/38 is accepted

  # Token set via ENV
}

variable "service" {
  description = "The name of your pager duty service"
}

variable "escalation_policy" {
  description = "The name of your escalation policy"
}

data "pagerduty_escalation_policy" "policy" {
  name = "${var.escalation_policy}"
}

resource pagerduty_service myservice {
  name              = "${var.service}"
  escalation_policy = "${data.pagerduty_escalation_policy.policy.id}"
  alert_creation    = "create_alerts_and_incidents"
}

resource "pagerduty_service_integration" "grafana" {
  name    = "Grafana API"
  type    = "events_api_v2_inbound_integration"
  service = "${pagerduty_service.myservice.id}"
}

resource "grafana_alert_notification" "pagerduty" {
  name       = "pagerduty for ${var.service}"
  type       = "pagerduty"
  is_default = false

  settings {
    "uploadImage"    = "true"
    "integrationKey" = "${pagerduty_service_integration.grafana.integration_key}"
    "autoResolve"    = "true"
  }
}

resource "grafana_dashboard" "dashboard" {
  config_json = "${file("grafana-dashboard.json")}"
}

output "grafana_dashboard" {
  value = "${grafana_dashboard.dashboard.id}"
}

output "pagerduty_service" {
  value = "${pagerduty_service.myservice.id}"
}

output "grafana_dashboard_url" {
  value = "https://grafana.internal.justin.tv/dashboard/db/${grafana_dashboard.dashboard.id}"
}

output "pagerduty_service_url" {
  value = "https://twitchoncall.pagerduty.com/services/${pagerduty_service.myservice.id}"
}
