package events

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/firehose"
	"github.com/aws/aws-sdk-go/service/firehose/firehoseiface"
)

// default limits for firehose
const (
	FirehoseBatchRecordCountLimit = 500
	FirehoseBatchSizeLimit        = 4 * (1024 ^ 2)
)

// DefaultReportInterval is 5 minutes
var DefaultReportInterval = 5 * time.Minute

// FirehoseBatchWriter batches data before writing to firehose
type FirehoseBatchWriter interface {
	WriteToFirehose(context.Context, []byte) error
}

// FirehoseBatcher batches records to upload to firehose
type FirehoseBatcher struct {
	BatchCount int
	MaxSize    int

	Firehose                   firehoseiface.FirehoseAPI
	FirehoseDeliveryStreamName string

	currentBatch     []*firehose.Record
	currentBatchSize int
}

func (fb *FirehoseBatcher) put(data []byte) {
	fb.ensureCurrentBatch()

	record := fb.formatData(data)
	fb.currentBatch = append(fb.currentBatch, record)
	fb.currentBatchSize += len(record.Data)
}

func (fb *FirehoseBatcher) ready() bool {
	return len(fb.currentBatch) >= fb.BatchCount || fb.currentBatchSize >= fb.MaxSize
}

func (fb *FirehoseBatcher) pop() (popped []*firehose.Record) {
	fb.ensureCurrentBatch()

	popped = fb.currentBatch
	poppedBatchSize := fb.currentBatchSize

	fb.currentBatch = nil
	fb.currentBatchSize = 0

	// trim the popped batch until it fits our parameters
	// must pop at least one
	for (len(popped) > fb.BatchCount || poppedBatchSize > fb.MaxSize) && len(popped) > 1 {
		toPutBack := popped[len(popped)-1]
		popped = popped[0 : len(popped)-1]
		poppedBatchSize -= len(toPutBack.Data)
		fb.unshiftRecord(toPutBack)
	}

	return
}

// unshift adds an entry to the front of the batch
func (fb *FirehoseBatcher) unshiftRecord(record *firehose.Record) {
	fb.ensureCurrentBatch()
	fb.currentBatch = append([]*firehose.Record{record}, fb.currentBatch...)
	fb.currentBatchSize += len(record.Data)
}

func (fb *FirehoseBatcher) ensureCurrentBatch() {
	if fb.currentBatch == nil {
		fb.currentBatch = make([]*firehose.Record, 0)
	}
}

func (fb *FirehoseBatcher) formatData(data []byte) *firehose.Record {
	return &firehose.Record{
		// Append \n to record before sending to firehose so athena
		// can read the records correctly.
		//see: https://forums.aws.amazon.com/thread.jspa?threadID=244858
		Data: append(data, []byte("\n")...),
	}
}

// WriteToFirehose buffers records in memory until batch is ready to write
func (fb *FirehoseBatcher) WriteToFirehose(ctx context.Context, data []byte) (err error) {
	if data == nil {
		return
	}

	if len(data) == 0 {
		return
	}

	fb.put(data)
	for {
		select {
		case <-ctx.Done():
			err = ctx.Err()
			return
		default:
		}

		if lastBatch := fb.pop(); len(lastBatch) > 0 {
			_, err = fb.Firehose.PutRecordBatchWithContext(ctx, &firehose.PutRecordBatchInput{
				Records:            lastBatch,
				DeliveryStreamName: aws.String(fb.FirehoseDeliveryStreamName),
			})
			if err != nil {
				return
			}
		} else {
			return
		}
	}
}
