package api

import (
	"errors"
	"net/http"
	"sort"
	"time"

	"goji.io/pat"

	"code.justin.tv/vod/vinyl/client"
	"code.justin.tv/web/channel-audits/client"
)

type LegendResponse struct {
	TotalTime   int              `json:"total_time"`
	Sessions    int              `json:"sessions"`
	LegendStart time.Time        `json:"legend_start"`
	LegendEnd   time.Time        `json:"legend_end"`
	GameSummary map[string]int64 `json:"game_summary"`
}

func (s *HTTPService) getLegend(r *http.Request) (interface{}, error) {
	userID := pat.Param(r, "user_id")

	vodResults, err := s.Vinyl.GetVodsByUser(r.Context(), userID, vinyl.GetVodsByUserInput{
		BroadcastType: "archive",
		Limit:         100,
		Sort:          "time",
		GetVodParams: &vinyl.GetVodParams{
			WithMuteInfo:             false,
			WithNotificationSettings: false,
			IncludeDeleted:           true,
			IncludePrivate:           true,
			IncludeProcessing:        true,
		},
	}, nil)

	if err != nil {
		return nil, err
	}

	vods := make([]*vinyl.Vod, len(vodResults.Vods))
	for i, j := 0, len(vodResults.Vods)-1; i < len(vodResults.Vods); i, j = i+1, j-1 {
		vods[i] = vodResults.Vods[j]
	}

	legendStart := time.Now()
	legendEnd := time.Now()
	totalTime := 0
	for _, vod := range vods {
		totalTime += vod.TotalLength
		if legendStart.After(vod.StartedOn) {
			legendStart = vod.StartedOn
		}
	}

	params := channel_audits.GetChannelAuditsParams{
		Channel: userID,
		Actions: []string{"game_change"},
		After:   0,
		Before:  time.Now().Unix(),
		Limit:   400,
	}
	audits, err := s.ChannelAudits.GetChannelAudits(r.Context(), params, nil)
	if err != nil {
		return nil, err
	}
	gameChanges := Audits(audits.ChannelAudits)
	sort.Sort(gameChanges)
	if len(gameChanges) == 0 {
		return nil, errors.New("No game changes.")
	}

	gameToTime := make(map[string]int64)
	gameChangeIdx := 0
	currentGame := gameChanges[0].OldValue
	for _, vod := range vods {
		for ; gameChangeIdx < len(gameChanges); gameChangeIdx++ {
			potentialGameChange := gameChanges[gameChangeIdx]
			currentGame = potentialGameChange.NewValue
			if potentialGameChange.CreatedOn.After(vod.StartedOn) {
				if potentialGameChange.OldValue != "" {
					currentGame = potentialGameChange.OldValue
				} else if gameChangeIdx > 0 {
					currentGame = gameChanges[gameChangeIdx-1].NewValue
				}
				gameChangeIdx -= 1
				break
			}
		}
		vodEnd := vod.StartedOn.Add(time.Duration(vod.TotalLength) * time.Second)
		breakPoint := vod.StartedOn
		for ; gameChangeIdx < len(gameChanges); gameChangeIdx++ {
			gameChange := gameChanges[gameChangeIdx]
			if gameChange.CreatedOn.After(vodEnd) {
				break
			}
			time := gameChange.CreatedOn.Unix() - breakPoint.Unix()
			gameToTime[currentGame] += time
			currentGame = gameChange.NewValue
			breakPoint = gameChange.CreatedOn
		}
		gameToTime[currentGame] += vodEnd.Unix() - breakPoint.Unix()
	}

	return LegendResponse{
		GameSummary: gameToTime,
		TotalTime:   totalTime,
		Sessions:    vodResults.TotalCount,
		LegendStart: legendStart.UTC(),
		LegendEnd:   legendEnd.UTC(),
	}, nil
}

type Audits []channel_audits.ChannelAudit

func (a Audits) Len() int           { return len(a) }
func (a Audits) Swap(i, j int)      { a[i], a[j] = a[j], a[i] }
func (a Audits) Less(i, j int) bool { return a[i].CreatedOn.Before(a[j].CreatedOn) }
